package ru.yandex.direct.jobs.cpaautobudget;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.autobudget.model.CpaAutobudgetPessimizedUser;
import ru.yandex.direct.core.entity.autobudget.repository.CpaAutobudgetPessimizedUsersYtRepository;
import ru.yandex.direct.core.entity.autobudget.service.CpaAutobudgetPessimizedUsersService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static java.time.temporal.ChronoUnit.MINUTES;
import static ru.yandex.direct.common.db.PpcPropertyNames.PAY_FOR_CONVERSION_PESSIMIZED_USERS_LAST_UPDATE_TIME;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.utils.DateTimeUtils.MOSCOW_TIMEZONE;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Обновление списка логинов пессимизированных cpa-автобюджетом
 * Данные поставляются ОКР через таблицу YT
 * В директе данные о логинах используются для показа предупреждений о недостатке конверсий
 * https://st.yandex-team.ru/DIRECT-122730
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 720),
        needCheck = ProductionOnly.class,

        //PRIORITY: ОКРная таблица апдейтится 1-2 раза в день. И эти данные влияют на предупреждения пользователям.
        // Поэтому в течение суток норм, но не дольше
        tags = {DIRECT_PRIORITY_1, CheckTag.DIRECT_PRODUCT_TEAM}
)
@Hourglass(periodInSeconds = 60 * 60 * 8, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class UpdatePayForConversionPessimizedUsersJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(UpdatePayForConversionPessimizedUsersJob.class);

    private static final int BATCH_SIZE = 5000;
    public static final int HOURS_TO_SUBTRACT_FOR_BOUNDARY_TIME = 6;
    private static final ZoneId zoneId = ZoneId.of(MOSCOW_TIMEZONE);

    private final CpaAutobudgetPessimizedUsersService cpaAutobudgetPessimizedUsersService;
    private final UserService userService;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final CpaAutobudgetPessimizedUsersYtRepository cpaAutobudgetPessimizedUsersYtRepository;

    @Autowired
    public UpdatePayForConversionPessimizedUsersJob(CpaAutobudgetPessimizedUsersService cpaAutobudgetPessimizedUsersService,
                                                    UserService userService,
                                                    PpcPropertiesSupport ppcPropertiesSupport,
                                                    CpaAutobudgetPessimizedUsersYtRepository cpaAutobudgetPessimizedUsersYtRepository) {
        this.cpaAutobudgetPessimizedUsersService = cpaAutobudgetPessimizedUsersService;
        this.userService = userService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.cpaAutobudgetPessimizedUsersYtRepository = cpaAutobudgetPessimizedUsersYtRepository;
    }

    @Override
    public void execute() {
        PpcProperty<LocalDateTime> lastModificationTimeProperty =
                ppcPropertiesSupport.get(PAY_FOR_CONVERSION_PESSIMIZED_USERS_LAST_UPDATE_TIME);
        LocalDateTime lastTableModificationTime = cpaAutobudgetPessimizedUsersYtRepository.getTableLastUpdateTime();

        if (!needUpdate(lastTableModificationTime, lastModificationTimeProperty)) {
            logger.info("Table has already been updated before. Skipping.");
            return;
        }

        int updatedRecordsCount = cpaAutobudgetPessimizedUsersYtRepository.getAndApplyPessimizedLogins(
                pessimizedLogins -> cpaAutobudgetPessimizedUsersService.addOrUpdatePessimizedLogins(
                        getUpdatedAndFilteredWithExistentClientId(pessimizedLogins)), BATCH_SIZE);
        logger.info("Processed {} records", updatedRecordsCount);

        //Все логины у которых updateTime старше now() считаем неактуальными и удаляем
        LocalDateTime boundaryDateTime =
                LocalDateTime.now(zoneId).truncatedTo(MINUTES).minusHours(HOURS_TO_SUBTRACT_FOR_BOUNDARY_TIME);
        cpaAutobudgetPessimizedUsersService.deleteLoginsWithUpdateTimeOlderThan(boundaryDateTime);

        lastModificationTimeProperty.set(lastTableModificationTime);
    }

    static boolean needUpdate(LocalDateTime lastTableModificationTime,
                              PpcProperty<LocalDateTime> lastModificationTimeProperty) {
        logger.info("{} property value is '{}'. Yt-table modification time is {}",
                PAY_FOR_CONVERSION_PESSIMIZED_USERS_LAST_UPDATE_TIME.getName(),
                lastModificationTimeProperty.get(), lastTableModificationTime);
        return lastTableModificationTime.isAfter(lastModificationTimeProperty.getOrDefault(LocalDateTime.MIN));
    }

    private List<CpaAutobudgetPessimizedUser> getUpdatedAndFilteredWithExistentClientId(List<CpaAutobudgetPessimizedUser> pessimizedLogins) {
        Map<String, Long> clientIdsByLogin = userService.massGetClientIdsByLogin(mapList(pessimizedLogins,
                CpaAutobudgetPessimizedUser::getLogin));

        if (clientIdsByLogin.size() != pessimizedLogins.size()) {
            var absentLogins = filterAndMapList(pessimizedLogins, row -> !clientIdsByLogin.containsKey(row.getLogin()),
                    CpaAutobudgetPessimizedUser::getLogin);
            logger.warn("Logins were not found and skipped: {}", absentLogins);
        }

        return filterAndMapList(pessimizedLogins,
                row -> clientIdsByLogin.containsKey(row.getLogin()),
                row -> row.withClientId(clientIdsByLogin.get(row.getLogin())));
    }
}
