package ru.yandex.direct.jobs.cpd;

import java.time.LocalDate;
import java.time.LocalTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesBannerRepository;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithTypeAndGeo;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.model.StatusPostModerate;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusPostModerate;
import ru.yandex.direct.core.entity.banner.type.moderation.BannerStatusRepository;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.model.CpmYndxFrontpageCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageCampaignShowType;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageBase;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackagesFilter;
import ru.yandex.direct.core.entity.pricepackage.model.ViewType;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.BaseDirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;

import static ru.yandex.direct.common.db.PpcPropertyNames.CPD_JOB_ENABLED;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;


@ParametersAreNonnullByDefault
@JugglerCheck(
        ttl = @JugglerCheck.Duration(hours = 8),
        tags = {DIRECT_PRIORITY_2},
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_ALEX_KULAKOV,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(cronExpression = "0 15 * * * ?", needSchedule = TypicalEnvironment.class)
public class CpdJob extends BaseDirectJob {

    private static final Logger logger = LoggerFactory.getLogger(CpdJob.class);
    private static final LocalTime TIME_TO_START_CAMPAIGNS = LocalTime.of(20, 0);

    PricePackageRepository pricePackageRepository;
    PricePackageService pricePackageService;
    CampaignTypedRepository campaignTypedRepository;
    ShardHelper shardHelper;
    GeoTreeFactory geoTreeFactory;
    YtProvider ytProvider;
    CampaignRepository campaignRepository;
    AdGroupRepository adGroupRepository;
    AggregatedStatusesBannerRepository aggregatedStatusesBannerRepository;
    BannerStatusRepository bannerStatusRepository;
    PpcPropertiesSupport ppcPropertiesSupport;


    @SuppressWarnings("checkstyle:ParameterNumber")
    public CpdJob(YtProvider ytProvider, PricePackageRepository pricePackageRepository,
                  PricePackageService pricePackageService, CampaignTypedRepository campaignTypedRepository,
                  ShardHelper shardHelper, GeoTreeFactory geoTreeFactory, CampaignRepository campaignRepository,
                  AdGroupRepository adGroupRepository,
                  AggregatedStatusesBannerRepository aggregatedStatusesBannerRepository,
                  BannerStatusRepository bannerStatusRepository, PpcPropertiesSupport ppcPropertiesSupport) {
        this.pricePackageRepository = pricePackageRepository;
        this.pricePackageService = pricePackageService;
        this.campaignTypedRepository = campaignTypedRepository;
        this.shardHelper = shardHelper;
        this.geoTreeFactory = geoTreeFactory;
        this.ytProvider = ytProvider;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.aggregatedStatusesBannerRepository = aggregatedStatusesBannerRepository;
        this.bannerStatusRepository = bannerStatusRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    private static ViewType frontpageCampaignShowTypeToViewType(FrontpageCampaignShowType frontpageCampaignShowType) {
        switch (frontpageCampaignShowType) {
            case FRONTPAGE:
                return ViewType.DESKTOP;
            case FRONTPAGE_MOBILE:
                return ViewType.MOBILE;
            case BROWSER_NEW_TAB:
                return ViewType.NEW_TAB;
            default:
                throw new RuntimeException("Не известный showType компании");
        }
    }

    private boolean isCampaignNeedToStop(List<BaseCampaign> cpdCampaigns, BaseCampaign campaignMaybeNeedToStop,
                                         List<AdGroupWithTypeAndGeo> groupsOfCpdCampaigns) {
        for (BaseCampaign cpdCampaign : cpdCampaigns) {
            if (isCampaignNeedToStopBecauseOfThisCpdCampaign(cpdCampaign, campaignMaybeNeedToStop)) {
                return isNeedToStopByGroups(campaignMaybeNeedToStop, groupsOfCpdCampaigns);
            }
        }
        return false;
    }

    private boolean isNeedToStopByGroups(BaseCampaign campaignMaybeNeedToStop,
                                         List<AdGroupWithTypeAndGeo> groupsOfCpdCampaigns) {
        int shard = shardHelper.getShardByCampaignId(campaignMaybeNeedToStop.getId());
        var groups = getGroupsByMapOFShardAndCampaignIds(Map.of(shard, List.of(campaignMaybeNeedToStop.getId())));
        if (groups.isEmpty()) {
            return false;
        }
        for (var group : groups) {
            if (!isThisGroupHasGeoIncludeInCpd(group, groupsOfCpdCampaigns)) {
                return false;
            }
        }
        return true;
    }


    private boolean isThisGroupHasGeoIncludeInCpd(AdGroupWithTypeAndGeo groupToCheck,
                                                  List<AdGroupWithTypeAndGeo> groupsOfCpdCampaigns) {
        GeoTree geoTree = geoTreeFactory.getGlobalGeoTree();
        for (var group : groupsOfCpdCampaigns) {
            if (geoTree.isOneRegionsIncludeAnotherRegions(group.getGeo(), groupToCheck.getGeo())) {
                return true;
            }
        }
        return false;
    }


    public boolean isThisGroupActive(Long groupId, int shard) {
        List<Long> bannerIds = aggregatedStatusesBannerRepository
                .getBannerIdsByAdGroupIds(shard, List.of(groupId));
        var bannersStatusInfo = bannerStatusRepository.getBannersStatusInfo(shard, bannerIds);
        for (var bannerStatusInfo : bannersStatusInfo) {
            if (bannerStatusInfo.getStatusModerate() == BannerStatusModerate.YES
                    && bannerStatusInfo.getStatusPostModerate().equals(BannerStatusPostModerate.YES)
                    && !bannerStatusInfo.isArchived()
                    && bannerStatusInfo.isStatusShow()
                    && bannerStatusInfo.isStatusActive()
                    && bannerStatusInfo.getAdGroupStatusModerate().equals(StatusModerate.YES)
                    && bannerStatusInfo.getAdGroupStatusPostModerate().equals(StatusPostModerate.YES)) {
                return true;
            }
        }
        return false;
    }

    private boolean isCampaignNeedToStopBecauseOfThisCpdCampaign(BaseCampaign cpdCampaign,
                                                                 BaseCampaign campaignMaybeNeedToStop) {
        if (cpdCampaign.getId() == campaignMaybeNeedToStop.getId()) {
            return false;
        }
        CpmPriceCampaign cpdTypedCampaign = (CpmPriceCampaign) cpdCampaign;
        var cpdViewTypes = cpdTypedCampaign.getFlightTargetingsSnapshot().getViewTypes();
        if (campaignMaybeNeedToStop instanceof CpmPriceCampaign) {
            CpmPriceCampaign typedCampaignMaybeNeedToStop = (CpmPriceCampaign) campaignMaybeNeedToStop;
            return cpdViewTypes.containsAll(typedCampaignMaybeNeedToStop.getFlightTargetingsSnapshot().getViewTypes());
        } else {
            CpmYndxFrontpageCampaign typedCampaignMaybeNeedToStop = (CpmYndxFrontpageCampaign) campaignMaybeNeedToStop;
            return cpdViewTypes.containsAll(
                    typedCampaignMaybeNeedToStop.getAllowedFrontpageType()
                            .stream()
                            .map(CpdJob::frontpageCampaignShowTypeToViewType)
                            .collect(Collectors.toList()));
        }
    }


    Map<Integer, List<? extends BaseCampaign>> getBaseCampaignsByShards(List<Long> campaignIds) {
        var campaignsIdsMapByShard = shardHelper.groupByShard(campaignIds, ShardKey.CID).getShardedDataMap();
        Map<Integer, List<? extends BaseCampaign>> result = new HashMap<>();
        campaignsIdsMapByShard.forEach((Integer shard, List<Long> campaigns) -> result.put(shard,
                campaignTypedRepository.getTypedCampaigns(shard, campaigns)));
        return result;
    }


    List<Long> getOnlyFrontPageCampaigns(List<Long> campaignsThatCouldBeFrontPageIds) {
        return campaignsThatCouldBeFrontPageIds.stream()
                .filter(cid -> {
                    var pricePackage = pricePackageService
                            .getPricePackageByCampaignIds(List.of(cid)).get(cid);
                    return pricePackage == null || pricePackage.isFrontpagePackage();
                })
                .collect(Collectors.toList());
    }


    List<Long> getCampaignsToStopInShard(Integer shard, List<Long> campaignIds, List<BaseCampaign> cpdCampaigns,
                                         List<AdGroupWithTypeAndGeo> groupsOfCpdCampaigns) {
        return campaignTypedRepository
                .getTypedCampaigns(shard, campaignIds).stream()
                .filter(campaign -> isCampaignNeedToStop(cpdCampaigns, campaign, groupsOfCpdCampaigns))
                .map(BaseCampaign::getId).collect(Collectors.toList());
    }

    Map<Integer, List<Long>> getCampaignsToStopByShard(Map<Integer, List<Long>> campaignsThatCouldBeFrontPageIdsByShard,
                                                       List<BaseCampaign> cpdCampaigns,
                                                       List<AdGroupWithTypeAndGeo> groupsOfCpdCampaigns) {
        campaignsThatCouldBeFrontPageIdsByShard.forEach((shard, campaignIds) -> campaignIds =
                getOnlyFrontPageCampaigns(campaignIds));


        return campaignsThatCouldBeFrontPageIdsByShard.keySet().stream()
                .collect(Collectors.toMap(Function.identity(),
                        shard -> getCampaignsToStopInShard(shard, campaignsThatCouldBeFrontPageIdsByShard.get(shard),
                                cpdCampaigns, groupsOfCpdCampaigns)));
    }

    List<BaseCampaign> getCpdBaseCampaigns(List<Long> cpdCampaignsIds) {
        var cpdCampaignsByShard = getBaseCampaignsByShards(cpdCampaignsIds);
        return cpdCampaignsByShard.values()
                .stream()
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
    }

    List<AdGroupWithTypeAndGeo> getGroupsByMapOFShardAndCampaignIds(Map<Integer, List<Long>> campaignsIdsByShard) {

        var groupsIdsByShard = campaignsIdsByShard.keySet()
                .stream()
                .collect(Collectors.toMap(Function.identity(),
                        shard -> (adGroupRepository
                                .getAdGroupIdsByCampaignIds(shard, campaignsIdsByShard.get(shard))).values()
                                .stream()
                                .flatMap(Collection::stream)
                                .filter(id -> isThisGroupActive(id, shard))
                                .collect(Collectors.toList())));
        return groupsIdsByShard.keySet()
                .stream().map(shard ->
                        adGroupRepository.getAdGroupsWithTypeAndGeo(shard,
                                        null,
                                        groupsIdsByShard.get(shard))
                                .values())
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
    }

    List<Long> getActivePackagesIdsWithCpd() {
        var packages = pricePackageRepository.getPricePackages(
                new PricePackagesFilter().withIsCpd(true),
                null,
                LimitOffset.maxLimited());
        return packages.getPricePackages().stream().filter(pricePackage ->
                        !pricePackage.getDateStart().isAfter(LocalDate.now())
                                && !pricePackage.getDateEnd().isBefore(LocalDate.now()))
                .map(PricePackageBase::getId).collect(Collectors.toList());
    }

    List<Long> getActiveCampaigns(int shard, Map<Long, Long> map) {
        List<Campaign> campaigns = campaignRepository.getCampaigns(shard, map.keySet());
        campaigns = campaigns.stream().filter(campaign -> campaign.getStartTime() != null &&
                campaign.getFinishTime() != null &&
                (campaign.getStartTime().isBefore(LocalDate.now())
                        || campaign.getStartTime().isEqual(LocalDate.now()))
                && (campaign.getFinishTime().isEqual(LocalDate.now())
                || campaign.getFinishTime().isAfter(LocalDate.now())) && !campaign.getStatusArchived()).collect(Collectors.toList());
        return campaigns.stream().map(Campaign::getId).collect(Collectors.toList());
    }

    private Map<Integer, List<Long>> getCpdCampaignsByShard(List<Long> packageIds) {
        Map<Integer, List<Long>> result = new HashMap<>();
        for (int shard : shardHelper.dbShards()) {
            var map = campaignRepository.getCampaignIdsByPricePackageIds(shard, packageIds);
            result.put(shard, getActiveCampaigns(shard, map));
        }
        return result;
    }

    Map<Integer, List<Long>> getCampaignsThatCouldBeFrontPageIdsByShard() {
        return shardHelper.dbShards().stream()
                .collect(Collectors.toMap(Function.identity(), shard ->
                        campaignRepository.getCampaignIdsThatCouldBeFronpageAndActive(shard, LocalDate.now())));
    }

    void stopedCampaigns(List<AdGroupWithTypeAndGeo> groupsOfCpdCampaigns, List<BaseCampaign> cpdCampaigns) {
        Map<Integer, List<Long>> campaignsThatCouldBeFrontPageIdsByShard = getCampaignsThatCouldBeFrontPageIdsByShard();
        var campaignsToStopByShard = getCampaignsToStopByShard(campaignsThatCouldBeFrontPageIdsByShard, cpdCampaigns,
                groupsOfCpdCampaigns);
        campaignsToStopByShard.forEach((Integer shard, List<Long> campaignIds)
                        -> {
                    if (!campaignIds.isEmpty()) {
                        campaignRepository.updateCampaignsIsCpdPaused(shard, campaignIds, true);
                    }
                }
        );
    }


    private void restartChangedCampaigns(List<AdGroupWithTypeAndGeo> groupsOfCpdCampaigns,
                                         List<BaseCampaign> cpdCampaigns) {
        Map<Integer, List<Long>> campaignMaybeNeedToRestart = new HashMap<>();
        for (int shard : shardHelper.dbShards()) {
            var campaignIdsByshard = campaignRepository
                    .getCampaignsWithIsCpdPaused(shard).stream()
                    .map(Campaign::getId)
                    .collect(Collectors.toList());
            campaignMaybeNeedToRestart.put(shard, campaignIdsByshard);
        }
        var campaignsToStopByShard = getCampaignsToStopByShard(campaignMaybeNeedToRestart, cpdCampaigns,
                groupsOfCpdCampaigns);
        for (int shard : shardHelper.dbShards()) {
            campaignMaybeNeedToRestart.get(shard).removeAll(campaignsToStopByShard.get(shard));
            if (campaignMaybeNeedToRestart.get(shard).isEmpty()) {
                continue;
            }
            campaignRepository.updateCampaignsIsCpdPaused(shard, campaignMaybeNeedToRestart.get(shard), false);
        }
    }

    void restartCampaigns() {
        for (int shard : shardHelper.dbShards()) {
            var campaignIdsToRestart = campaignRepository
                    .getCampaignsWithIsCpdPaused(shard).stream()
                    .map(Campaign::getId)
                    .collect(Collectors.toList());
            if (!campaignIdsToRestart.isEmpty()) {
                campaignRepository.updateCampaignsIsCpdPaused(shard, campaignIdsToRestart, false);
            }
        }
    }

    private void stopCampaignsAndRestartChangedCampaigns() {
        var packageIds = getActivePackagesIdsWithCpd();
        if (packageIds.isEmpty()) {
            restartCampaigns();
            logger.info("Нет ни одного cpd пакета, все остановленные компании перезапущены");
            return;
        }
        var cpdCampaignsIdsByShard = getCpdCampaignsByShard(packageIds);
        var groupsOfCpdCampaigns = getGroupsByMapOFShardAndCampaignIds(cpdCampaignsIdsByShard);
        if (groupsOfCpdCampaigns.isEmpty()) {
            restartCampaigns();
            logger.info("Нет ни одной активной cpd группы, все остановленные компании перезапущены");
            return;
        }
        var cpdCampaigns = getCpdBaseCampaigns(cpdCampaignsIdsByShard.values()
                .stream()
                .flatMap(Collection::stream)
                .collect(Collectors.toList()));
        if (cpdCampaigns.isEmpty()) {
            restartCampaigns();
            logger.info("Нет ни одной cpd компании, все остановленные компании перезапущены");
            return;
        }
        stopedCampaigns(groupsOfCpdCampaigns, cpdCampaigns);
        logger.info("Компании остановлены");
        restartChangedCampaigns(groupsOfCpdCampaigns, cpdCampaigns);
        logger.info("Измененные компании больше не попадающие под условия перезапущены");
    }


    @Override
    public void execute() {
        var isNeedToWork = ppcPropertiesSupport.get(CPD_JOB_ENABLED).getOrDefault(false);
        if (!isNeedToWork) {
            logger.info("Джоба пропускается, так как CPD_JOB_ENABLED = false");
            return;
        }
        try {
            if (LocalTime.now().isAfter(TIME_TO_START_CAMPAIGNS)) {
                restartCampaigns();
                logger.info("Все компании перезапущены");
            } else {
                stopCampaignsAndRestartChangedCampaigns();
            }
        } catch (Exception e) {
            logger.error("Произошла ошибка, джоба не отработала: " + e.getMessage());
        }
    }
}


