package ru.yandex.direct.jobs.crypta

import com.google.common.base.Preconditions.checkState
import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.IMPORT_CRYPTA_SEGMENTS_FOR_CA_ENABLED
import ru.yandex.direct.core.entity.crypta.model.CaCryptaSegmentExport
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentsYtRepository
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalScope
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.env.TypicalEnvironment
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import java.lang.IllegalArgumentException


/**
 * Импортирует в таблицу {@link Tables#CRYPTA_GOALS} новые сегменты для использования в Custom Audience из YT таблицы:
 * //home/crypta/production/profiles/export/lab/direct_exports_tanker_names
 * Джоба только добавляет записи! Удаление или обновление не делается
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(days = 1, hours = 12),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_2, CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_KUVSHINOV],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
@Hourglass(cronExpression = "0 0 9-17/4 * * *", needSchedule = TypicalEnvironment::class)
class ImportCryptaSegmentsForCaJob(
    private val cryptaSegmentRepository: CryptaSegmentRepository,
    private val cryptaSegmentsYtRepository: CryptaSegmentsYtRepository,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
) : DirectJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(ImportCryptaSegmentsForCaJob::class.java)
    }

    private val isJobEnabled
        get() = ppcPropertiesSupport.get(IMPORT_CRYPTA_SEGMENTS_FOR_CA_ENABLED)
            .getOrDefault(false)

    override fun execute() {
        if (!isJobEnabled) {
            logger.info("Job is disabled")
            return
        }

        val cryptaSegments = cryptaSegmentsYtRepository.getAll()
        checkState(cryptaSegments.isNotEmpty(), "fetched records from YT can't be empty")
        logger.info("Fetched ${cryptaSegments.size} rows from YT")

        val directCryptaGoals = cryptaSegmentRepository.getAll(
            listOf(CryptaGoalScope.PERFORMANCE, CryptaGoalScope.MEDIA)
        )
        logger.info("Fetched ${directCryptaGoals.size} rows from crypta_goals")

        val directSegments = CryptaSegmentsConverter.convertGoalsToSegments(directCryptaGoals.values)
        val lastId = directCryptaGoals.keys.maxOrNull() ?: (Goal.CRYPTA_INTERESTS_CA_LOWER_BOUND - 1)

        val newCryptaGoals = cryptaSegments.filter {
            !it.exports.isNullOrEmpty() && !directContainsSegment(directSegments, it.exports)
        }.mapIndexedNotNull { i, value ->
            CryptaSegmentsConverter.convertYtDataToGoal(value, lastId + i + 1)
        }

        if (newCryptaGoals.isEmpty()) {
            logger.info("Nothing to import. Table crypta_goals is up to date")
            return
        }

        val addedCount = cryptaSegmentRepository.add(newCryptaGoals)

        logger.info("Added $addedCount new rows")
    }

    private fun directContainsSegment(
        directSegments: Set<CaCryptaSegmentExport>,
        cryptaSegments: List<CaCryptaSegmentExport>
    ) = when (cryptaSegments.size) {
        1 -> directSegments.contains(cryptaSegments[0])
        // crypta cannot change segments
        2 -> directSegments.contains(cryptaSegments[0]) || directSegments.contains(cryptaSegments[1])
        else -> throw IllegalArgumentException(
            "Crypta exports can't have more than 2 elements: " + cryptaSegments.joinToString { it.toString() }
        )
    }
}
