package ru.yandex.direct.jobs.dialogs;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.campaign.model.Dialog;
import ru.yandex.direct.core.entity.dialogs.service.DialogsService;
import ru.yandex.direct.dialogs.client.model.Skill;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TestingOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static java.lang.Integer.min;
import static ru.yandex.direct.common.db.PpcPropertyNames.ENABLE_SYNC_CAMP_DIALOGS_STATUS;

/**
 *  Обновление статуса чатов. Делает запрос в Диалоги, статус меняется на неактивный в следующих случаях:
 *  - ручка говорит что такого skillId вообще нет
 *  - ручка говорит что у чата значение свойства botGuid == null
 *  - ручка говорит что у чата значение свойства onAir == false
 *  Запускаем каждые 20 минут с 2 до 4 ночи
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 2, hours = 4),
        needCheck = NonDevelopmentEnvironment.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_MAXLOG,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}))
@Hourglass(cronExpression = "0 0/20 2,3 * * ?", needSchedule = ProductionOnly.class)
@Hourglass(cronExpression = "0 0/20 10-18 * * ?", needSchedule = TestingOnly.class)
public class CampDialogsIsActiveSyncJob extends DirectShardedJob {
    private final DialogsService dialogsService;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    private static final int CHUNK_SIZE = 100; // Обратываем по 100 кампании за раз
    private static final int HOURS_BETWEEN_SYNC = 22;

    private static final Logger logger = LoggerFactory.getLogger(CampDialogsIsActiveSyncJob.class);

    @Autowired
    public CampDialogsIsActiveSyncJob(
            DialogsService dialogsService,
            PpcPropertiesSupport ppcPropertiesSupport
    ) {
        this.dialogsService = dialogsService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    // этот конструктор используется для теста
    CampDialogsIsActiveSyncJob(
            int shard,
            DialogsService dialogsService,
            PpcPropertiesSupport ppcPropertiesSupport) {
        super(shard);
        this.dialogsService = dialogsService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    @Override
    public void execute() {
        if (!isJobEnabled()) {
            logger.info("Skip processing. Job is not enabled.");
            return;
        }

        syncDialogsByChunk(CHUNK_SIZE, HOURS_BETWEEN_SYNC);
    }

    void syncDialogsByChunk(int chunkSize, int hoursBetweenSync) {
        int shard = getShard();
        int offset = 0;
        int processedSkillCount = 0;
        LocalDateTime syncOverThan = LocalDateTime.now().minusHours(hoursBetweenSync);
        List<Dialog> dialogs = dialogsService.getDialogs(shard, syncOverThan);
        List<String> skillIds = StreamEx.of(dialogs)
                .map(Dialog::getSkillId)
                .distinct()
                .toList();
        do {
            int fromIndex = min(offset, skillIds.size());
            int toIndex = min(offset + chunkSize, skillIds.size());
            List<String> skillsIdsToSync = skillIds.subList(fromIndex, toIndex);
            Map<String, Skill> skillByIds =  StreamEx.of(dialogsService.getSkills(skillsIdsToSync))
                    .toMap(Skill::getSkillId, Function.identity());
            offset += chunkSize;
            try {
                dialogsService.updateDialogsBySkills(shard, dialogs, skillByIds);
                processedSkillCount += skillByIds.size();
            } catch (RuntimeException e) {
                logger.error("Error while sync skills", e);
                setJugglerStatus(JugglerStatus.WARN,
                        String.format("Error on processing chunk with %d dialogs", chunkSize));
            }
        } while (offset < skillIds.size());
        logger.info("sync {} skills", processedSkillCount);
    }

    private boolean isJobEnabled() {
        return ppcPropertiesSupport.get(ENABLE_SYNC_CAMP_DIALOGS_STATUS).getOrDefault(false);
    }
}
