package ru.yandex.direct.jobs.directdb;

import java.sql.SQLException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.impl.DefaultSetF;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.jobs.directdb.exception.HomeDirectDbJobException;
import ru.yandex.direct.jobs.directdb.metrics.HomeDirectDbMetricsReporter;
import ru.yandex.direct.jobs.directdb.metrics.HomeDirectDbOperationsMetricProvider;
import ru.yandex.direct.jobs.directdb.model.Operation;
import ru.yandex.direct.jobs.directdb.model.SnapshotAttributes;
import ru.yandex.direct.jobs.directdb.repository.OperationRepository;
import ru.yandex.direct.jobs.directdb.service.HomeDirectDbFullWorkPropObtainerService;
import ru.yandex.direct.jobs.directdb.service.SnapshotUserAttributeService;
import ru.yandex.direct.jobs.directdb.service.YqlClasspathObtainerService;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.exceptions.RuntimeYqlException;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yql.ResultSetFuture;
import ru.yandex.yql.YqlConnection;
import ru.yandex.yql.YqlDataSource;

import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.common.db.PpcPropertyNames.HOME_DIRECT_DB_DATE;
import static ru.yandex.direct.common.db.PpcPropertyNames.HOME_DIRECT_DB_EXCLUDE;
import static ru.yandex.direct.common.db.PpcPropertyNames.HOME_DIRECT_DB_INCLUDE;
import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;

/**
 * Джоб, который запускает YQL-запросы, которые находятся в classpath:resources/export/home-direct-db для создания
 * аггрегаций над снепшотами снятыми с //home/direct/mysql-sync
 * <p>
 * Параметры:
 * командной строки:
 * - кластер
 * из ppcproperties:
 * - date (HOME_DIRECT_DB_DATE) - дата за которую будет запускаться джоб. От нее зависит какой снепшот будет выбран
 * для запуска операций, если они еще не запущены, остальные части джоба от даты не зависят (трекинг статусов,
 * перезапуск)
 * - include (HOME_DIRECT_DB_INCLUDE) - только запросы, которые находятся в этом множестве будут запущены на новом
 * снепшоте, если параметр не задан или список пустой, то будут запускаться все запросы, кроме тех, которые указаны в
 * параметре ниже
 * - exclude (HOME_DIRECT_DB_EXCLUDE) - эти запросы будут исключены из списка запуска
 * - fullWork (HOME_DIRECT_DB_FULL_WORK) - выполнять всю работу или нет (если нет, то запросы не запускаются и джоба
 * работает в режиме "dry run")
 * <p>
 * Джоб состоит из 4 частей:
 * - Запуск YQL'ей
 * - Проверка статусов
 * - Перезапуск упавших операций
 * - Создание симлинка
 * <p>
 * Подробнее про каждую стадию выполнения джоба:
 * <p>
 * Запуск YQL'ей:
 * Сперва мы формируем список запросов, берем из classpath:resources/export/home-direct-db
 * пары - название -> контент. В контенте запроса перед самим запросом еще добавляется include/common.yql, в
 * котором содержатся общие переменные и функции, нужные для всех YQL'ей. При запуске запросов используем
 * prepared statements для подстановки в запрос путей, по которым располагаются снепшоты и //home/direct/db.
 * Расположение новых таблиц происходит в следующем формате: //home/direct/db-archive/YYYY-MM-DD/[table_name],
 * где YYYY-MM-DD - дата снятия снепшота mysql-sync, table_name - название результирующей таблицы, получившейся в
 * результате аггрегации. Это позволяет нам из коробки получить архивы.
 * После запуска пишем в сервисную дин таблицу operations метаинформацию о запущенных операциях: operationId для
 * восстановления Future и проверки статуса, время старта, время окончания, дату снепшота, статус, сообщение об
 * ошибке и т.д. Здесь есть потенциальная точка отказа, если упадет на этом этапе, то при следующем запуске,
 * запросы будут запущены снова, с точки зрения корректности ничего плохого, но YT-кластер будет делать лишнюю
 * работу. Таблица operations - динамическая и для обращения к ней используется API для динтаблиц, потому что после
 * того, как мы запустим кучу запросов под нашей квотой, количество параллельно выполняемых операций будет
 * исчерпано и мы не сможем в фоне обновлять статусы, к тому же, по динтаблице будет быстрый lookup и данных в ней
 * мало.
 * <p>
 * Проверка статусов:
 * Вытаскиваем все операции из служебной таблицы за последнюю неделю от текущей даты, у которых статус не DONE,
 * восстанавливаем Future по operationId и пытаемся взять, если взяли, то операция завершена, если нет, в сообщении
 * Exception'а будет содержаться ошибка, которая случилась на кластере, обновляем статусы перезапущенных задач,
 * инкрементим attempts и пишем ошибку в errorMessage, чтобы даже если операция запущена и выполняется, была
 * возможность узнать, что с операцией было не так, пишем обратно в сервисную таблицу.
 * <p>
 * Перезапуск упавших операций:
 * Тот же список операций, взятый после проверки статусов фильтруется на наличие операций со статусом ERROR, далее,
 * проверяется, что количество attempts < MAX_ATTEMPTS, если нет, то перезапускаем операцию, иначе требуется помощь
 * человека и перезапускать смысла нет.
 * Как происходит перезапуск: просто снова запускается
 * {@link HomeDirectDbOperationsJob#runOperation(YtCluster, SnapshotAttributes, LocalDate, Pair)}, который генерит
 * нам новый operationId, который надо записать в таблицу operations.
 * <p>
 * Создание симлинка:
 * Симлинк создается с последнего //home/direct/db-archive/YYYY-MM-DD на //home/direct/db-archive/current, снепшот
 * считается обработанным, когда все взятые за одну дату операции будут иметь статус DONE, вычисляется это на
 * основе таблицы operations. Так как в таблице operations за определенную дату снепшота могут быть либо все
 * запускавшиеся операции либо ни одной, мы можем безопасно вычислять стасус готовности снепшота.
 * Если симлинка не было создано до этого этапа, или в результате работы симлинк будет изменен, тогда мы можем
 * посчитать качество данных.
 *
 * @see "https://st.yandex-team.ru/DIRECT-110792"
 */
@ParametersAreNonnullByDefault
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 30 + 1), needCheck = ProductionOnly.class, tags = {DIRECT_PRIORITY_1})
@ParameterizedBy(parametersSource = DirectExportYtClustersParametersSource.class)
@Hourglass(periodInSeconds = 60 * 15)
public class HomeDirectDbOperationsJob extends DirectParameterizedJob<YtCluster> {
    private static final Logger logger = LoggerFactory.getLogger(HomeDirectDbOperationsJob.class);

    private static final String DB_PATH = "db-archive";
    private static final int MAX_ATTEMPTS = 5;
    private static final String SYMLINK_TARGET_PATH_ATTRIBUTE_NAME = "target_path";

    private final DirectExportYtClustersParametersSource parametersSource;
    private final SnapshotUserAttributeService snapshotUserAttributeService;
    private final YqlClasspathObtainerService yqlClasspathObtainerService;
    private final YtProvider ytProvider;
    private final ShardHelper shardHelper;
    private final OperationRepository operationRepository;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final HomeDirectDbOperationsMetricProvider homeDirectDbOperationsMetricProvider;
    private final HomeDirectDbMetricsReporter homeDirectDbMetricsReporter;
    private final HomeDirectDbFullWorkPropObtainerService fullWorkPropObtainerService;

    @SuppressWarnings("squid:S00107")
    public HomeDirectDbOperationsJob(
            DirectExportYtClustersParametersSource parametersSource,
            SnapshotUserAttributeService snapshotUserAttributeService,
            YqlClasspathObtainerService yqlClasspathObtainerService,
            YtProvider ytProvider,
            ShardHelper shardHelper,
            OperationRepository operationRepository,
            PpcPropertiesSupport ppcPropertiesSupport,
            HomeDirectDbOperationsMetricProvider homeDirectDbOperationsMetricProvider,
            HomeDirectDbMetricsReporter homeDirectDbMetricsReporter,
            HomeDirectDbFullWorkPropObtainerService fullWorkPropObtainerService
    ) {
        this.parametersSource = parametersSource;
        this.snapshotUserAttributeService = snapshotUserAttributeService;
        this.yqlClasspathObtainerService = yqlClasspathObtainerService;
        this.ytProvider = ytProvider;
        this.shardHelper = shardHelper;
        this.operationRepository = operationRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.homeDirectDbOperationsMetricProvider = homeDirectDbOperationsMetricProvider;
        this.homeDirectDbMetricsReporter = homeDirectDbMetricsReporter;
        this.fullWorkPropObtainerService = fullWorkPropObtainerService;
    }

    @Override
    public void execute() {
        YtCluster cluster = parametersSource.convertStringToParam(getParam());

        logger.info("Getting job date from ppc properties: {}", HOME_DIRECT_DB_DATE.getName());
        LocalDate jobDate = getJobDate();

        logger.info("Getting listing of mysql-sync directory with attributes of each element");
        Collection<SnapshotAttributes> attributesCollection =
                snapshotUserAttributeService.getConvertedUserAttributes(cluster);
        logger.info("Attributes are {}", attributesCollection);

        logger.info("Obtaining queries from classpath");
        Collection<Pair<String, String>> queries = yqlClasspathObtainerService.obtainYqlQueriesFromClassPath();
        Map<String, String> queryMap = queries.stream().collect(toMap(Pair::getKey, Pair::getValue));
        logger.info("Query list is {}", queryMap.keySet());

        Optional<SnapshotAttributes> lastSnapshot = attributesCollection
                .stream()
                .filter(snapshot -> snapshot.getCypressPath().contains(jobDate.toString()))
                .filter(SnapshotAttributes::isFinished)
                .findFirst();
        lastSnapshot.ifPresent(snapshot -> handleSnapshot(cluster, jobDate, queries, snapshot));

        logger.info("Obtaining not finished operations");
        Collection<Operation> notFinishedOperations = operationRepository.getNotFinished(cluster, jobDate);

        logger.info("Checking statuses of {}", notFinishedOperations);
        checkStatuses(cluster, notFinishedOperations);

        logger.info("Rerunning errored operations");
        lastSnapshot.ifPresent(snapshot -> reRunErrored(cluster, notFinishedOperations, queryMap, snapshot));

        logger.info("Looking up the latest finished snapshot");
        operationRepository.getLatestReadySnapshotDate(cluster).ifPresent(date -> makeSymLink(cluster, date));

        logger.info("Sending metrics");
        homeDirectDbOperationsMetricProvider.loadAndSendMetrics(cluster);
        homeDirectDbMetricsReporter.report(cluster);
    }

    /**
     * Получение даты из ppc properties.
     * <p>
     * Сайд-эффект: сброс даты в ppc properties, если она была установлена, это необходимо для того, чтобы этот
     * параметр не влиял на последующие запуски.
     *
     * @return дата из ppcproperties или сегодняшняя, если там такой записи не оказалось или она сброшена
     */
    private LocalDate getJobDate() {
        Optional<LocalDate> maybeJobDate = Optional.ofNullable(ppcPropertiesSupport.get(HOME_DIRECT_DB_DATE).get());
        maybeJobDate.ifPresent(jobDate -> {
            logger.info("Date exists in ppc property {}, its value is {}", HOME_DIRECT_DB_DATE.getName(), jobDate);
            ppcPropertiesSupport.remove(HOME_DIRECT_DB_DATE.getName());
        });

        LocalDate now = LocalDate.now();
        if (maybeJobDate.isEmpty()) {
            logger.info(
                    "Date does not exist in ppc property {}, obtaining current date: {}",
                    HOME_DIRECT_DB_DATE.getName(),
                    now
            );
        }
        return maybeJobDate.orElse(now);
    }

    private void handleSnapshot(YtCluster cluster,
                                LocalDate jobDate,
                                Collection<Pair<String, String>> queries,
                                SnapshotAttributes snapshot) {
        logger.info("Snapshot {} is finished", snapshot.getCypressPath());

        logger.info("Getting operations for date {}", jobDate);
        Collection<Operation> operationsForDate = operationRepository.getForDate(cluster, jobDate);

        logger.info("Running operations for {}: {}", jobDate, operationsForDate);
        runOperations(cluster, jobDate, queries, snapshot, operationsForDate);
    }

    private void makeSymLink(YtCluster cluster, LocalDate symlinkDate) {
        String home = getHome(cluster);
        String rawCurrent = YtPathUtil.generatePath(home, relativePart(), DB_PATH, "current");
        String rawLastReady = YtPathUtil.generatePath(home, relativePart(), DB_PATH, symlinkDate.toString());
        YPath current = YPath.simple(rawCurrent);
        YPath lastReady = YPath.simple(rawLastReady);

        boolean isSymlinkWillBeChanged = checkSymlinkWillBeChanged(cluster, rawLastReady, current);

        if (isSymlinkWillBeChanged) {
            logger.info("Creating symlink {} -> {}", rawLastReady, rawCurrent);
            ytProvider.get(cluster).cypress().link(lastReady, current, true);
        }
    }

    private boolean checkSymlinkWillBeChanged(YtCluster cluster, String rawLastReady, YPath current) {
        Cypress cypress = ytProvider.get(cluster).cypress();

        if (!cypress.exists(YPath.simple(rawLastReady))) {
            logger.warn("Path {} does not exist, symlink won't be changed", rawLastReady);
            return false;
        }

        if (cypress.exists(current)) {
            YTreeNode yTreeNode = cypress.get(
                    YPath.simple(YtPathUtil.generatePath(getHome(cluster), relativePart(), DB_PATH)),
                    new DefaultSetF<>(singleton(SYMLINK_TARGET_PATH_ATTRIBUTE_NAME))
            );
            Optional<String> maybeTargetPath = yTreeNode
                    .mapNode()
                    .get("current")
                    .flatMap(node -> node.getAttribute(SYMLINK_TARGET_PATH_ATTRIBUTE_NAME))
                    .map(YTreeNode::stringValue);

            boolean result = false;
            if (maybeTargetPath.isPresent() && !maybeTargetPath.get().equals(rawLastReady)) {
                logger.info("Symlink will be changed from {} to {}", maybeTargetPath.get(), rawLastReady);
                result = true;
            }

            return result;
        } else {
            logger.info("Symlink will be created to {}", rawLastReady);
            return true;
        }
    }

    private String getHome(YtCluster cluster) {
        return ytProvider.getClusterConfig(cluster).getHome();
    }

    @SuppressWarnings("squid:S1192")
    private void runOperations(YtCluster cluster,
                               LocalDate jobDate,
                               Collection<Pair<String, String>> queries,
                               SnapshotAttributes snapshot,
                               Collection<Operation> operations) {
        logger.info(
                "Getting queries included for {} from {}",
                snapshot.getCypressPath(),
                HOME_DIRECT_DB_INCLUDE.getName()
        );
        Set<String> includeQueries = getQuerySetFromPpcProperties(HOME_DIRECT_DB_INCLUDE);
        logger.info("Queries included for {} are {}", snapshot.getCypressPath(), includeQueries);
        logger.info(
                "Getting queries excluded for {} from {}",
                snapshot.getCypressPath(),
                HOME_DIRECT_DB_EXCLUDE.getName()
        );
        Set<String> excludeQueries = getQuerySetFromPpcProperties(HOME_DIRECT_DB_EXCLUDE);
        logger.info("Queries excluded for {} are {}", snapshot.getCypressPath(), excludeQueries);

        if (!includeQueries.isEmpty() && !excludeQueries.isEmpty()) {
            logger.error(
                    "Only include or exclude should be set, not both. Include: {} Exclude: {}",
                    includeQueries,
                    excludeQueries
            );
            throw new IllegalArgumentException("Only INCLUDE or EXCLUDE ppc properties should be set, not both");
        }

        if (operations.isEmpty()) {
            logger.info("Operations are empty, should run queries for snapshot {}", snapshot.getCypressPath());

            if (!fullWorkPropObtainerService.isFullWorkEnabled()) {
                logger.info("Full work is disabled, do not run queries");
                return;
            }

            List<Operation> newOperations = queries
                    .stream()
                    .filter(query -> includeQueries.isEmpty() || includeQueries.contains(query.getKey()))
                    .filter(query -> !excludeQueries.contains(query.getKey()))
                    .map(query -> runOperation(cluster, snapshot, jobDate, query))
                    .filter(Optional::isPresent)
                    .map(Optional::get)
                    .collect(Collectors.toList());
            logger.info("Writing operations {} to operations meta table", newOperations);
            operationRepository.upsert(cluster, newOperations);
        }
    }

    private Set<String> getQuerySetFromPpcProperties(PpcPropertyName<Set<String>> ppcProperty) {
        return Optional
                .ofNullable(ppcPropertiesSupport.get(ppcProperty).get())
                .orElse(Collections.emptySet())
                .stream()
                .map(query -> query + ".yql")
                .collect(toSet());
    }

    private void checkStatuses(YtCluster cluster, Collection<Operation> operations) {
        operations
                .stream()
                .filter(operation -> operation.getStatus() == Operation.OperationStatus.IN_PROGRESS)
                .forEach(operation -> updateStatus(cluster, operation));
        logger.info("Writing operations {} to operations meta table", operations);
        operationRepository.upsert(cluster, operations);
    }

    private void reRunErrored(YtCluster cluster, Collection<Operation> operations, Map<String, String> queryMap,
                              SnapshotAttributes snapshot) {
        operations
                .stream()
                .filter(op -> isNeedToRetry(op, getSnapshotDateString(snapshot)))
                .forEach(query -> reRunOperation(query, cluster, snapshot, queryMap));
        logger.info("Writing operations {} to operations meta table", operations);
        operationRepository.upsert(cluster, operations);
    }

    private Optional<Operation> runOperation(YtCluster cluster,
                                             SnapshotAttributes snapshot,
                                             LocalDate jobDate,
                                             Pair<String, String> query) {
        String name = query.getKey();
        String snapshotPath = snapshot.getCypressPath();

        YtOperator ytOperator = ytProvider.getOperator(cluster, YtSQLSyntaxVersion.SQLv1);

        final ResultSetFuture future;
        logger.info("Running query {} asynchronously for snapshot {}", name, snapshotPath);

        try {
            future = ytOperator.yqlQueryBegin(
                    new YqlQuery(query.getValue(), shardHelper.dbShards().size(), snapshotPath,
                            getQueryOutputTablePath(cluster, getSnapshotDateString(snapshot), name))
                            .withTitle(String.format("HomeDirectDbOperationsJob.%s.%s", name, jobDate)));
        } catch (RuntimeYqlException e) {
            return Optional.empty();
        }

        String operationId = future.getOperationId();
        logger.info("Operation id is {} for query {} for snapshot {}", operationId, name, snapshotPath);

        Optional<Operation> operation = Optional.of(
                new Operation(
                        name,
                        LocalDateTime.now(),
                        null,
                        jobDate,
                        operationId,
                        Operation.OperationStatus.IN_PROGRESS,
                        1,
                        null
                )
        );

        logger.info("Operation meta is {} for query {} for snapshot {}", operation, name, snapshotPath);

        return operation;
    }

    private String getQueryOutputTablePath(YtCluster cluster, String date, String queryFileName) {
        String tableName = Pattern.compile("\\.yql", Pattern.CASE_INSENSITIVE)
                .matcher(queryFileName).replaceAll("");
        return YtPathUtil.generatePath(getHome(cluster), relativePart(), DB_PATH, date, tableName);
    }

    private String getSnapshotDateString(SnapshotAttributes snapshot) {
        int firstOccurrence = snapshot.getCypressPath().indexOf("--");
        int secondOccurrence = snapshot.getCypressPath().indexOf("--", firstOccurrence + 1);
        return snapshot.getCypressPath().substring(secondOccurrence + 2);
    }

    private void updateStatus(YtCluster cluster, Operation operation) {
        logger.info("Checking status of operation {}", operation);
        YqlDataSource yql = ytProvider.getYql(cluster, YtSQLSyntaxVersion.SQLv1);

        try (YqlConnection yqlConnection = (YqlConnection) yql.getConnection()) {
            logger.info("Restoring future from operation with id {}", operation.getOperationId());
            Future<?> future = yqlConnection.restoreResultSetFuture(operation.getOperationId(), "");
            if (future.isDone()) {
                logger.info("Operation {} is done, trying to get result", operation.getOperationId());
                future.get();
                logger.info("Result of operation {} was successfully got", operation.getOperationId());

                setTableUpdateTime(operation.getName(), operation.getDate(), cluster);
                operation.setStatus(Operation.OperationStatus.DONE);
                operation.setFinish(LocalDateTime.now());
            } else {
                logger.info("Operation {} is in progress", operation.getOperationId());
                operation.setStatus(Operation.OperationStatus.IN_PROGRESS);
            }
        } catch (SQLException e) {
            throw new HomeDirectDbJobException(e);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } catch (ExecutionException e) {
            logger.warn(String.format(
                    "There is an error obtained when operation %s restored",
                    operation.getOperationId()),
                    e
            );
            operation.setStatus(Operation.OperationStatus.ERROR);
            operation.setErrorMessage(e.toString());
        }
    }

    private void setTableUpdateTime(String operationName, LocalDate operationDate, YtCluster cluster) {
        YPath outputTablePath = YPath.simple(getQueryOutputTablePath(cluster, operationDate.toString(), operationName));
        YPath uploadTimeAttr = outputTablePath.attribute("upload_time");
        String uploadTime = operationDate.atStartOfDay(MSK).withNano(0).format(DateTimeFormatter.ISO_INSTANT);
        ytProvider.get(cluster).cypress().set(uploadTimeAttr, uploadTime);
    }

    private void reRunOperation(Operation operation, YtCluster cluster, SnapshotAttributes snapshot,
                                Map<String, String> queryMap) {
        logger.warn("Retrying operation {}, error message: {}", operation, operation.getErrorMessage());

        String query = queryMap.get(operation.getName());
        Optional<Operation> maybeOperation = runOperation(
                cluster,
                snapshot,
                operation.getDate(),
                Pair.of(operation.getName(), query)
        );
        maybeOperation.ifPresent(newOperation -> {
            String operationId = newOperation.getOperationId();
            logger.info("Operation {} successfully retried, new operationId is {}", operation, operationId);

            operation.setOperationId(operationId);
            operation.setStatus(Operation.OperationStatus.IN_PROGRESS);

            logger.info("Incrementing attempts for operation {} to {}", operation, operation.getAttempts() + 1);
            operation.setAttempts(operation.getAttempts() + 1); // Safe because one thread
        });
    }

    private boolean isNeedToRetry(Operation operation, String snapshotDate) {
        boolean isShouldBeRetried =
                operation.getStatus() == Operation.OperationStatus.ERROR && operation.getAttempts() < MAX_ATTEMPTS
                        && operation.getDate().toString().equals(snapshotDate);
        if (isShouldBeRetried) {
            logger.info("Operation {} should be retried", operation);
        } else {
            logger.info("Operation {} should not be retried", operation);
        }
        return isShouldBeRetried;
    }
}
