package ru.yandex.direct.jobs.directdb.metrics;

import java.time.ZonedDateTime;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.jobs.directdb.service.YqlClasspathObtainerService;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

@Component
@ParametersAreNonnullByDefault
public class HomeDirectDbMetricsReporter {

    private static final String NEW_DB_YT_NODE = "//home/direct/db-archive/current";

    private final YqlClasspathObtainerService yqlClasspathObtainerService;
    private final YtProvider ytProvider;
    private final SolomonPushClient solomonPushClient;

    HomeDirectDbMetricsReporter(
            YtProvider ytProvider,
            YqlClasspathObtainerService yqlClasspathObtainerService,
            SolomonPushClient solomonPushClient
    ) {
        this.yqlClasspathObtainerService = yqlClasspathObtainerService;
        this.ytProvider = ytProvider;
        this.solomonPushClient = solomonPushClient;
    }

    public void report(YtCluster cluster) {
        MetricRegistry metricsRegistry =
                SolomonUtils.newPushRegistry("flow", "yt_direct_db", "yt_cluster", cluster.getName());
        registerExportDelayMetrics(metricsRegistry, cluster);
        solomonPushClient.sendMetrics(metricsRegistry);
    }

    /**
     * Рассчитывает возраст самой старой таблицы в выгрузке.
     */
    private void registerExportDelayMetrics(MetricRegistry metricsRegistry, YtCluster cluster) {
        List<String> queryNames = yqlClasspathObtainerService.getYqlQueryNames();
        long newExportAge = ZonedDateTime.now().toEpochSecond()
                - ZonedDateTime.parse(getOldestTableInYtNode(NEW_DB_YT_NODE, cluster, queryNames)).toEpochSecond();
        metricsRegistry.gaugeInt64("new_export_age_seconds").set(newExportAge);
    }

    /**
     * Проходит по всем таблицам YT ноды с именами `tableNames`, и выясняет, какой "возраст" у данных в этой таблице.
     * Для вычисления возраста берётся минимальная строка из атрибутов "creation_time" и "upload_time"
     */
    private String getOldestTableInYtNode(String nodePath, YtCluster cluster, List<String> tableNames) {
        Cypress cypress = ytProvider.get(cluster).cypress();
        Map<String, YTreeNode> creationTimes = cypress
                .get(YPath.simple(nodePath), Cf.set("creation_time", "upload_time"))
                .asMap();
        return tableNames.stream()
                .map(creationTimes::get)
                .filter(v -> v != null)
                .map(table -> minStr(
                        table.getAttributeOrThrow("creation_time").stringValue(),
                        table.getAttributeOrThrow("upload_time").stringValue()
                ))
                .sorted()
                .findFirst().orElseThrow();
    }

    private static String minStr(String a, String b) {
        if (a.compareTo(b) < 1) {
            return a;
        }
        return b;
    }
}
