package ru.yandex.direct.jobs.directdb.metrics;

import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Collection;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.jobs.directdb.model.Operation;
import ru.yandex.direct.jobs.directdb.repository.OperationRepository;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * Считывает метрики для {@link ru.yandex.direct.jobs.directdb.HomeDirectDbOperationsJob} и отправляет их в Solomon
 */
@Component
@ParametersAreNonnullByDefault
public class HomeDirectDbOperationsMetricProvider {
    private static final Logger logger = LoggerFactory.getLogger(HomeDirectDbOperationsMetricProvider.class);

    private final OperationRepository operationRepository;
    private final SolomonPushClient solomonPushClient;

    private static class Metrics {
        /**
         * Количество запущенных операций
         */
        private final long running;
        /**
         * Количество завершенных операций
         */
        private final long done;
        /**
         * Количество упавших операций
         */
        private final long failed;
        /**
         * Количество операций, у которых attempts > 1
         */
        private final long withErrors;
        /**
         * Количество времени прошедшее со старта первой операции до завершения последней (в минутах)
         */
        private final long durationInMinutes;

        /**
         * Timestamp последнего собранного архива
         */
        private final long lastGatheredTimestamp;

        @SuppressWarnings("squid:S00107")
        public Metrics(long running,
                       long done,
                       long failed,
                       long withErrors,
                       long durationInMinutes,
                       long lastGatheredTimestamp) {
            this.running = running;
            this.done = done;
            this.failed = failed;
            this.withErrors = withErrors;
            this.durationInMinutes = durationInMinutes;
            this.lastGatheredTimestamp = lastGatheredTimestamp;
        }

        public long getRunning() {
            return running;
        }

        public long getDone() {
            return done;
        }

        public long getFailed() {
            return failed;
        }

        public long getWithErrors() {
            return withErrors;
        }

        public Long getDurationInMinutes() {
            return durationInMinutes;
        }

        public Long getLastGatheredTimestamp() {
            return lastGatheredTimestamp;
        }
    }

    public HomeDirectDbOperationsMetricProvider(OperationRepository operationRepository,
            SolomonPushClient solomonPushClient) {
        this.operationRepository = operationRepository;
        this.solomonPushClient = solomonPushClient;
    }

    public void loadAndSendMetrics(YtCluster cluster) {
        logger.info("Collecting metrics for new //home/direct/db-archive (cluster: {})", cluster);
        var now = LocalDate.now();
        var operations = getOperationsForDate(cluster, now);

        var metricRegistry = SolomonUtils.newPushRegistry(
                "yt_cluster", cluster.getName(),
                "flow", "home_direct_db_operations"
        );

        sendMetrics(metricRegistry, new Metrics(
                filterOperationsInStatus(operations, Operation.OperationStatus.IN_PROGRESS),
                filterOperationsInStatus(operations, Operation.OperationStatus.DONE),
                filterOperationsInStatus(operations, Operation.OperationStatus.ERROR),
                filterErroredOperations(operations),
                getLastDurationOfGathering(cluster),
                getLatestGatheredTimestamp(cluster)
        ), cluster);
    }

    private Long getLatestGatheredTimestamp(YtCluster cluster) {
        logger.info("Collecting last gathered archive timestamp");
        var lastGatheredTimestamp = operationRepository
                .getLatestGatheredTimestamp(cluster)
                .map(datetime -> datetime.atZone(ZoneId.systemDefault()).toEpochSecond())
                .orElse(0L);
        logger.info("Collected last gathered archive timestamp: {}", lastGatheredTimestamp);
        return lastGatheredTimestamp;
    }

    private Long getLastDurationOfGathering(YtCluster cluster) {
        logger.info("Collecting duration");
        var durationInMinutes = operationRepository.getLastDurationOfGathering(cluster).orElse(0L);
        logger.info("Collected duration: {}", durationInMinutes);
        return durationInMinutes;
    }

    private long filterErroredOperations(Collection<Operation> operations) {
        logger.info("Collecting errored operations");
        var withErrors = operations.stream().filter(operation -> operation.getAttempts() > 1).count();
        logger.info("Collected {} errored operations", withErrors);
        return withErrors;
    }

    private long filterOperationsInStatus(Collection<Operation> operations, Operation.OperationStatus status) {
        logger.info("Collecting operations in status {}", status);
        var filteredOperations = operations
                .stream()
                .filter(operation -> operation.getStatus() == status)
                .count();
        logger.info("Collected operations {}", filteredOperations);
        return filteredOperations;
    }

    private Collection<Operation> getOperationsForDate(YtCluster cluster, LocalDate now) {
        logger.info("Obtaining all operations for {}", now);
        var operations = operationRepository.getForDate(cluster, now);
        logger.info("Obtained operations: {}", operations);
        return operations;
    }

    private void sendMetrics(MetricRegistry metricRegistry, Metrics metric, YtCluster cluster) {
        logger.info("Registering sensors for cluster {}", cluster);
        logger.info("Registering home_direct_db_operations_run");
        metricRegistry
                .gaugeInt64("home_direct_db_operations_run")
                .set(metric.getRunning());
        logger.info("Registering home_direct_db_operations_done");
        metricRegistry
                .gaugeInt64("home_direct_db_operations_done")
                .set(metric.getDone());
        logger.info("Registering home_direct_db_operations_failed");
        metricRegistry
                .gaugeInt64("home_direct_db_operations_failed")
                .set(metric.getFailed());
        logger.info("Registering home_direct_db_operations_with_errors");
        metricRegistry
                .gaugeInt64("home_direct_db_operations_with_errors")
                .set(metric.getWithErrors());
        logger.info("Registering home_direct_db_operations_duration_in_minutes");
        metricRegistry
                .gaugeInt64("home_direct_db_operations_duration_in_minutes")
                .set(metric.getDurationInMinutes());
        logger.info("Registering home_direct_db_operations_last_gathered_timestamp");
        metricRegistry
                .gaugeInt64("home_direct_db_operations_last_gathered_timestamp")
                .set(metric.getLastGatheredTimestamp());

        solomonPushClient.sendMetrics(metricRegistry);
    }

}
