package ru.yandex.direct.jobs.directdb.repository;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.impl.EmptyMap;
import ru.yandex.direct.jobs.directdb.model.Operation;
import ru.yandex.direct.jobs.directdb.service.YqlClasspathObtainerService;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeBooleanNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YtTables;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.inside.yt.kosher.tables.YTableEntryTypes.YSON;

@Repository
@ParametersAreNonnullByDefault
public class OperationRepository {
    private static final Logger logger = LoggerFactory.getLogger(OperationRepository.class);

    private static final String TABLE_NAME = "operations";
    private static final String TABLE_PATH = "db-archive";
    private static final String QUERY_TEMPLATE_FOR_DATE =
            "name, date, operationId, start, finish, attempts, status, errorMessage FROM [%s] WHERE date = '%s'";
    private static final String QUERY_TEMPLATE_FOR_NOT_FINISHED =
            "name, date, operationId, start, finish, attempts, status, errorMessage " +
            "FROM [%s] WHERE status != 'DONE' AND date > '%s'";
    private static final String QUERY_FOR_DETERMINING_LATEST_FINISHED_DATE =
            "date, 1 AS temp, sum(temp) AS doneNumber FROM [%s] WHERE status = 'DONE' GROUP BY date " +
                    " HAVING doneNumber = %d ORDER BY date DESC LIMIT 1";

    private static final String NAME_KEY = "name";
    private static final String TYPE_KEY = "type";
    private static final String STRING_TYPE = "string";
    private static final String SORT_ORDER_KEY = "sort_order";
    private static final String ASC_SORT_ORDER = "ascending";
    private static final String INT_64_TYPE = "int64";

    private static final String FINISH_FIELD_NAME = "finish";
    private static final String OPERATION_ID_FIELD_NAME = "operationId";
    private static final String STATUS_FIELD_NAME = "status";
    private static final String START_FIELD_NAME = "start";
    private static final String DATE_FIELD_NAME = "date";
    private static final String ATTEMTPS_FIELD_NAME = "attempts";
    private static final String ERROR_MESSAGE_FIELD_NAME = "errorMessage";
    private static final String NAME_FIELD_NAME = "name";

    // за сколько последних дней берем незавершенные операции
    private static final int PERIOD_IN_DAYS_FOR_NOT_FINISHED_OPERATIONS = 7;

    private final YtProvider ytProvider;
    private final YqlClasspathObtainerService yqlClasspathObtainerService;

    @Autowired
    public OperationRepository(YtProvider ytProvider, YqlClasspathObtainerService yqlClasspathObtainerService) {
        this.ytProvider = ytProvider;
        this.yqlClasspathObtainerService = yqlClasspathObtainerService;
    }

    public Collection<Operation> getNotFinished(YtCluster cluster, LocalDate jobData) {
        String operationsTablePath = getOperationsTablePath(cluster);
        createDynamicTableIfDoesNotExist(operationsTablePath, cluster);

        String query = String.format(QUERY_TEMPLATE_FOR_NOT_FINISHED, operationsTablePath,
                jobData.minusDays(PERIOD_IN_DAYS_FOR_NOT_FINISHED_OPERATIONS));
        return selectOperations(cluster, query);
    }

    public Collection<Operation> getForDate(YtCluster cluster, LocalDate date) {
        String operationsTablePath = getOperationsTablePath(cluster);
        createDynamicTableIfDoesNotExist(operationsTablePath, cluster);

        String query = String.format(QUERY_TEMPLATE_FOR_DATE, operationsTablePath, date.toString());
        return selectOperations(cluster, query);
    }

    /**
     * Получить время работы всех операций за один день.
     * <p>
     * Значение не точное, потому как finish ставится тогда, когда будет запущен job и он возьмет статусы всех
     * YQL-операций, тогда, если он понимает, что операция завершена, тогда ставит finish текущим временем, но этой
     * точности будет достаточно, чтобы понимать примерное время работы запросов.
     *
     * @param cluster кластер YT
     * @return количество минут затраченных на выполнение всех запросов
     */
    public Optional<Long> getLastDurationOfGathering(YtCluster cluster) {
        return getLatestReadySnapshotDate(cluster)
                .flatMap(lastFinishedDate -> getLastDurationOfGathering(cluster, lastFinishedDate));
    }

    private Optional<? extends Long> getLastDurationOfGathering(YtCluster cluster, LocalDate lastFinishedDate) {
        Collection<Operation> operations = getForDate(cluster, lastFinishedDate);

        Optional<LocalDateTime> maybeMin = operations.stream().map(Operation::getStart).min(Comparator.naturalOrder());
        Optional<LocalDateTime> maybeMax = operations.stream().map(Operation::getFinish).max(Comparator.naturalOrder());

        return maybeMax.flatMap(max -> maybeMin.map(min -> Duration.between(min, max).toMinutes()));
    }

    public Optional<LocalDateTime> getLatestGatheredTimestamp(YtCluster cluster) {
        Optional<LocalDate> latestReadySnapshotDate = getLatestReadySnapshotDate(cluster);
        return latestReadySnapshotDate
                .flatMap(
                        date -> getForDate(cluster, date)
                                .stream()
                                .map(Operation::getFinish)
                                .max(Comparator.naturalOrder())
                );
    }

    public Optional<LocalDate> getLatestReadySnapshotDate(YtCluster cluster) {
        int numberOfQueries = yqlClasspathObtainerService.obtainYqlQueriesFromClassPath().size();

        String operationsTablePath = getOperationsTablePath(cluster);
        createDynamicTableIfDoesNotExist(operationsTablePath, cluster);

        YtTables tables = ytProvider.get(cluster).tables();
        AtomicReference<Optional<LocalDate>> result = new AtomicReference<>(Optional.empty());
        String query = String.format(QUERY_FOR_DETERMINING_LATEST_FINISHED_DATE, operationsTablePath, numberOfQueries);
        tables.selectRows(
                query,
                YSON,
                (YTreeMapNode row) -> result
                        .set(row.getStringO(DATE_FIELD_NAME).map(LocalDate::parse))
        );

        Optional<LocalDate> maybeSnapshotDate = result.get();
        if (maybeSnapshotDate.isPresent()) {
            logger.info("Latest finished snapshot date found {}", maybeSnapshotDate.get());
        } else {
            logger.info("Latest finished snapshot not found");
        }
        return maybeSnapshotDate;
    }

    public void upsert(YtCluster cluster, Collection<Operation> operations) {
        String operationsTablePath = getOperationsTablePath(cluster);
        createDynamicTableIfDoesNotExist(operationsTablePath, cluster);

        YtTables tables = ytProvider.get(cluster).tables();
        List<YTreeMapNode> ytOperations = operations.stream().map(this::buildYTreeNode).collect(Collectors.toList());
        tables.insertRows(YPath.simple(operationsTablePath), true, false, YSON, Cf.wrap(ytOperations).iterator());
    }

    private String getDbPath(YtCluster cluster) {
        String home = ytProvider.getClusterConfig(cluster).getHome();
        return YtPathUtil.generatePath(home, relativePart(), TABLE_PATH);
    }

    private String getOperationsTablePath(YtCluster cluster) {
        String home = ytProvider.getClusterConfig(cluster).getHome();
        return YtPathUtil.generatePath(home, relativePart(), TABLE_PATH, TABLE_NAME);
    }

    private List<Operation> selectOperations(YtCluster cluster, String query) {
        YtTables tables = ytProvider.get(cluster).tables();
        List<Operation> operations = new ArrayList<>();
        tables.selectRows(query, YSON, (YTreeMapNode row) -> operations.add(mapYsonToOperation(row)));
        return operations;
    }

    private Operation mapYsonToOperation(YTreeMapNode row) {
        LocalDateTime finish;
        try {
            finish = LocalDateTime.parse(row.getString(FINISH_FIELD_NAME));
        } catch (DateTimeParseException e) {
            finish = null;
        }
        return mapYsonToOperation(row, finish);
    }

    private Operation mapYsonToOperation(YTreeMapNode row, @Nullable LocalDateTime finish) {
        String name = row.getString(NAME_FIELD_NAME);
        String operationId = row.getString(OPERATION_ID_FIELD_NAME);
        Operation.OperationStatus status = Operation.OperationStatus.valueOf(row.getString(STATUS_FIELD_NAME));
        LocalDateTime start = LocalDateTime.parse(row.getString(START_FIELD_NAME));
        LocalDate date = LocalDate.parse(row.getString(DATE_FIELD_NAME));
        int attempts = row.getInt(ATTEMTPS_FIELD_NAME);
        String errorMessage = row.getString(ERROR_MESSAGE_FIELD_NAME);
        return new Operation(name, start, finish, date, operationId, status, attempts, errorMessage);
    }

    private void createDynamicTableIfDoesNotExist(String operationsTablePath, YtCluster cluster) {
        Cypress cypress = ytProvider.get(cluster).cypress();
        String dbPath = getDbPath(cluster);
        if (!cypress.exists(YPath.simple(dbPath))) {
            cypress.create(YPath.simple(dbPath), CypressNodeType.MAP);
        }
        if (!cypress.exists(YPath.simple(operationsTablePath))) {
            logger.info("Operations table was not exist, creating");
            createDynamicTable(operationsTablePath, cluster);
        } else {
            logger.info("Operations table exists, do nothing");
        }
    }

    private void createDynamicTable(String operationsTablePath, YtCluster cluster) {
        Cypress cypress = ytProvider.get(cluster).cypress();
        YtTables tables = ytProvider.get(cluster).tables();
        Map<String, YTreeNode> attributes = getDynamicSchema();
        cypress.create(YPath.simple(operationsTablePath), CypressNodeType.TABLE, attributes);
        tables.mount(YPath.simple(operationsTablePath));
    }

    private Map<String, YTreeNode> getDynamicSchema() {
        return Map.of(
                "dynamic", new YTreeBooleanNodeImpl(true, new EmptyMap<>()),
                YtUtils.SCHEMA_ATTR, YTree
                        .listBuilder()
                        .value(
                                YTree
                                        .mapBuilder()
                                        .key(NAME_KEY).value(NAME_FIELD_NAME)
                                        .key(TYPE_KEY).value(STRING_TYPE)
                                        .key(SORT_ORDER_KEY).value(ASC_SORT_ORDER)
                                        .buildMap()
                        )
                        .value(
                                YTree
                                        .mapBuilder()
                                        .key(NAME_KEY).value(DATE_FIELD_NAME)
                                        .key(TYPE_KEY).value(STRING_TYPE)
                                        .key(SORT_ORDER_KEY).value(ASC_SORT_ORDER)
                                        .buildMap()
                        )
                        .value(
                                YTree
                                        .mapBuilder()
                                        .key(NAME_KEY).value(OPERATION_ID_FIELD_NAME)
                                        .key(TYPE_KEY).value(STRING_TYPE)
                                        .buildMap()
                        )
                        .value(
                                YTree
                                        .mapBuilder()
                                        .key(NAME_KEY).value(START_FIELD_NAME)
                                        .key(TYPE_KEY).value(STRING_TYPE)
                                        .buildMap()
                        )
                        .value(
                                YTree
                                        .mapBuilder()
                                        .key(NAME_KEY).value(FINISH_FIELD_NAME)
                                        .key(TYPE_KEY).value(STRING_TYPE)
                                        .buildMap()
                        )
                        .value(
                                YTree
                                        .mapBuilder()
                                        .key(NAME_KEY).value(ATTEMTPS_FIELD_NAME)
                                        .key(TYPE_KEY).value(INT_64_TYPE)
                                        .buildMap()
                        )
                        .value(
                                YTree
                                        .mapBuilder()
                                        .key(NAME_KEY).value(STATUS_FIELD_NAME)
                                        .key(TYPE_KEY).value(STRING_TYPE)
                                        .buildMap()
                        )
                        .value(
                                YTree
                                        .mapBuilder()
                                        .key(NAME_KEY).value(ERROR_MESSAGE_FIELD_NAME)
                                        .key(TYPE_KEY).value(STRING_TYPE)
                                        .buildMap()
                        )
                        .buildList()
        );
    }

    private YTreeMapNode buildYTreeNode(Operation operation) {
        return YTree
                .mapBuilder()
                .key(START_FIELD_NAME).value(operation.getStart().toString())
                .key(FINISH_FIELD_NAME).value(operation.getFinish() == null ? "" :
                        operation.getFinish().toString())
                .key(ATTEMTPS_FIELD_NAME).value(operation.getAttempts())
                .key(DATE_FIELD_NAME).value(operation.getDate().toString())
                .key(ERROR_MESSAGE_FIELD_NAME).value(operation.getErrorMessage() == null ? "" :
                        operation.getErrorMessage())
                .key(NAME_FIELD_NAME).value(operation.getName())
                .key(OPERATION_ID_FIELD_NAME).value(operation.getOperationId())
                .key(STATUS_FIELD_NAME).value(operation.getStatus().toString())
                .buildMap();
    }

}
