package ru.yandex.direct.jobs.directdb.service;

import java.time.LocalDate;
import java.time.format.DateTimeParseException;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.bolts.collection.impl.EmptyMap;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.exceptions.OperationRunningException;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.DataSize;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeBooleanNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeIntegerNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeStringNodeImpl;
import ru.yandex.inside.yt.kosher.operations.Operation;
import ru.yandex.inside.yt.kosher.operations.YtOperations;
import ru.yandex.inside.yt.kosher.operations.specs.JobIo;
import ru.yandex.inside.yt.kosher.operations.specs.MergeMode;
import ru.yandex.inside.yt.kosher.operations.specs.MergeSpec;
import ru.yandex.inside.yt.kosher.tables.TableWriterOptions;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;
import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;

import static java.lang.Math.max;
import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;

/**
 * Пережимаем таблицы, которые старше 7 дней, более эффективным кодеком.
 * <p>
 * <p>
 * Когда джоб запускается, он в каждой папке, которая старше, чем 7 дней (называние папки в //home/direct/db-archive -
 * это должна быть либо дата, либо симлинк current) мы просматриваем аттрибуты всех таблиц для того, чтобы понять надо
 * ли что-то делать или нет.
 * <p>
 * Как мы определяем, что надо что-то делать: если compression_codec стоит отличный от
 * {@link HomeDirectDbArchivingService#COMPRESSION_CODEC_VALUE}, тогда, в надежде, что никто не поставит его таким
 * "руками", запускаем транзакцию, в транзакции, меняем ему атрибуты (compression_codec, erasure_codec) и запускаем
 * Merge-операцию (https://wiki.yandex-team.ru/yt/userdoc/erasure) в этой же транзакции, чтобы, если операция
 * завершится неуспешно, compression_codec сбросился и, когда job придет в следующий раз, он понял, что на этой таблице
 * нужно выполнять архивацию.
 * <p>
 * В транзакциях YT подразумевается активное взаимодействие клиента с кластером, нужно постоянно пинговать транзакцию,
 * мы пингуем раз в 5 секунд, так как используем стандартный таймаут транзакции (15 секунд). Полагаемся на то,
 * в нормальных условиях это будет работать без перебоев, но если случится GC-пауза, например, то транзакция просто
 * абортится на кластере и мы снова придем на этот узел и попытаемся снова заархивировать эту тблицу
 */
@Service
@ParametersAreNonnullByDefault
public class HomeDirectDbArchivingService {

    private static final Logger logger = LoggerFactory.getLogger(HomeDirectDbArchivingService.class);

    private static final int DAYS_BEFORE_ARCHIVING = 7;
    private static final long DESIRED_CHUNK_SIZE = (long) (4 * 1024) * (4 * 1024) * (4 * 1024);
    private static final int DEFAULT_DATA_SIZE_PER_JOB = 256 * 1024 * 1024;
    private static final String HOME_DIRECT_DB_PATH = "db-archive";
    private static final String COMPRESSION_CODEC_VALUE = "brotli_8";
    private static final String ERASURE_CODEC_VALUE = "lrc_12_2_2";
    private static final String COMPRESSION_CODEC_KEY = "compression_codec";
    private static final String ERASURE_CODEC_KEY = "erasure_codec";
    private static final String SORTED_KEY = "sorted";
    private static final String SORT_BY_KEY = "sort_by";
    private static final String FORCE_TRANSFORM_KEY = "force_transform";

    private final YtProvider ytProvider;
    private final HomeDirectDbFullWorkPropObtainerService fullWorkPropObtainerService;

    // Флаг для остановки работы сервиса. В данном случае его полезно использовать, так как джоба состоит из
    // последовательного архивирования большого количества таблиц. При этом промежуточный стейт джобы
    // сохраняется "из коробки" - у таблиц в YT выставляется атрибут, по которому определяется ее архивированность.
    // Запускать архивирование в несколько потоков выглядит пока что слишком расточительным: джоба работает
    // не больше 3 часов и активно выполняется не больше одного раща в сутки (когда наступает новый день).
    private volatile boolean isShutdown = false;

    public HomeDirectDbArchivingService(YtProvider ytProvider,
                                        HomeDirectDbFullWorkPropObtainerService fullWorkPropObtainerService
    ) {
        this.ytProvider = ytProvider;
        this.fullWorkPropObtainerService = fullWorkPropObtainerService;
    }

    public void archive(YtCluster cluster, LocalDate now) {
        String home = ytProvider.getClusterConfig(cluster).getHome();
        String foldersPath = YtPathUtil.generatePath(home, relativePart(), HOME_DIRECT_DB_PATH);

        YPath folders = YPath.simple(foldersPath);
        List<YTreeStringNode> list = ytProvider
                .get(cluster)
                .cypress()
                .list(folders)
                .stream()
                .sorted(Comparator.comparing(YTreeStringNode::stringValue))
                .collect(Collectors.toList());

        for (YTreeStringNode node : list) {
            logger.info("handling node {}", node);
            handleNode(cluster, foldersPath, now, node);
        }
    }

    private void handleNode(YtCluster cluster, String foldersPath, LocalDate now, YTreeStringNode node) {
        String folderName = node.stringValue();
        try {
            handleNode(cluster, foldersPath, now, folderName);
        } catch (DateTimeParseException e) {
            logger.info("{} is not a snapshot", folderName);
        }
    }

    private void handleNode(YtCluster cluster, String foldersPath, LocalDate now, String folderName) {
        LocalDate nodeDate = LocalDate.parse(folderName);

        logger.info("Checking should we archive {}", folderName);
        if (!nodeDate.isBefore(now.minusDays(DAYS_BEFORE_ARCHIVING))) {
            logger.info("We should not archive {}", folderName);
            return;
        }

        logger.info("We should archive {}", folderName);
        handleFolder(cluster, foldersPath, folderName);
    }

    private void handleFolder(YtCluster cluster, String foldersPath, String folderName) {
        List<YTreeStringNode> tables = ytProvider
                .get(cluster)
                .cypress()
                .list(YPath.simple(YtPathUtil.generatePath(foldersPath, folderName)))
                .stream()
                .sorted(Comparator.comparing(YTreeStringNode::stringValue))
                .collect(Collectors.toList());

        for (YTreeStringNode table : tables) {
            if (isShutdown) {
                logger.info("Shutting down at node {}/{}", folderName, table);
                return;
            }
            logger.info("Archiving node {}", table);
            handleTable(cluster, foldersPath, folderName, table);
        }
    }

    private void handleTable(YtCluster cluster, String foldersPath, String folderName, YTreeStringNode table) {
        Yt yt = ytProvider.get(cluster);
        Cypress cypress = yt.cypress();
        YtOperations operations = yt.operations();

        String tableName = table.stringValue();
        String tablePath = YtPathUtil.generatePath(foldersPath, folderName, tableName);
        Map<String, YTreeNode> attributes = cypress
                .get(YPath.simple(YtPathUtil.generatePath(tablePath, "@")))
                .asMap();

        Optional.ofNullable(attributes.get(COMPRESSION_CODEC_KEY)).ifPresent(compressionCodec -> {
            logger.info("Compression codec for {} is {}", tablePath, compressionCodec.stringValue());
            if (compressionCodec.stringValue().equals(COMPRESSION_CODEC_VALUE)) {
                logger.info("Table {} already compressed", tablePath);
                return;
            }

            if (!fullWorkPropObtainerService.isFullWorkEnabled()) {
                logger.info("Full work is disabled, skip compressing for {}", tablePath);
                return;
            }

            logger.info("Start compressing {}", tablePath);

            YtDynamicOperator ytDynamicOperator = ytProvider.getDynamicOperator(cluster);
            ApiServiceTransactionOptions transactionOptions = new ApiServiceTransactionOptions(ETransactionType.TT_MASTER)
                    .setSticky(true)
                    .setPing(true);

            try {
                ytDynamicOperator.runInTransaction(tx -> {
                    GUID guid = tx.getId();

                    setCypressArchiveAttributes(cypress, tablePath, guid);

                    MergeSpec mergeSpec = getMergeSpec(tablePath, attributes);
                    Operation operation = operations.mergeAndGetOp(Optional.of(guid), false, mergeSpec);
                    operation.await();
                }, transactionOptions);
            } catch (OperationRunningException e) {
                logger.warn("Transaction was aborted", e);
            }

            logger.info("Finish compressing {}", tablePath);
        });
    }

    private MergeSpec getMergeSpec(String tablePath, Map<String, YTreeNode> attributes) {
        MergeSpec.Builder mergeSpecBuilder = MergeSpec.builder()
            .setInputTables(Collections.singletonList(YPath.simple(tablePath)))
            .setOutputTable(YPath.simple(tablePath));

        logger.info("Getting data size per job for {}", tablePath);
        int dataSizePerJob = getDataSizePerJob(attributes);
        logger.info("Data size per job for {} is {}", tablePath, dataSizePerJob);

        YTreeBooleanNodeImpl yTreeTrue = new YTreeBooleanNodeImpl(true, new EmptyMap<>());
        YTreeIntegerNodeImpl yTreeDataSizePerJob = new YTreeIntegerNodeImpl(false, dataSizePerJob, new EmptyMap<>());
        TableWriterOptions tableWriter = new TableWriterOptions().withDesiredChunkSize(DataSize.fromBytes(DESIRED_CHUNK_SIZE));
        JobIo jobIo = new JobIo(tableWriter);

        logger.info("Getting merge mode for {}", tablePath);
        MergeMode mergeMode = getMergeMode(attributes);
        logger.info("Merge mode for {} is {}", tablePath, mergeMode);

        return mergeSpecBuilder
                .setMergeMode(mergeMode)
                .setCombineChunks(true)
                .setAdditionalSpecParameters(Map.of(
                        FORCE_TRANSFORM_KEY, yTreeTrue,
                        YtUtils.DATA_SIZE_PER_JOB_ATTR, yTreeDataSizePerJob
                ))
                .setJobIo(jobIo)
                .build();
    }

    private int getDataSizePerJob(Map<String, YTreeNode> attributes) {
        Optional<YTreeNode> maybeCompressionRatio = Optional.ofNullable(attributes.get(YtUtils.COMPRESSION_RATIO_ATTR));
        if (maybeCompressionRatio.isPresent()) {
            double compressionRatio = maybeCompressionRatio.get().doubleValue();
            logger.info("Compression ratio is {}", compressionRatio);

            // Очень странное место, calculatedDataSizePerJob почти всегда равен 2^31-1, так как результат
            // деления не влезает в int. Возможно это не ошибка, и так действительно было задумано, так как это
            // значение проверяется в тестах.
            int calculatedDataSizePerJob = (int) (DESIRED_CHUNK_SIZE / compressionRatio);
            return max(DEFAULT_DATA_SIZE_PER_JOB, calculatedDataSizePerJob);
        }
        logger.info("Using default data size per job");
        return DEFAULT_DATA_SIZE_PER_JOB;
    }

    private MergeMode getMergeMode(Map<String, YTreeNode> attributes) {
        Optional<YTreeNode> maybeSorted = Optional.ofNullable(attributes.get(SORTED_KEY));

        boolean wellSorted = maybeSorted.isPresent() && maybeSorted.get().boolValue();

        Optional<YTreeNode> maybeSortby = Optional.ofNullable(attributes.get(SORT_BY_KEY));
        wellSorted = wellSorted || maybeSortby.isPresent();

        return wellSorted ? MergeMode.SORTED : MergeMode.UNORDERED;
    }

    private void setCypressArchiveAttributes(Cypress cypress, String tablePath, GUID transactionId) {
        cypress.set(
                Optional.of(transactionId),
                false,
                YPath.simple(YtPathUtil.generatePath(tablePath, "@" + COMPRESSION_CODEC_KEY)),
                new YTreeStringNodeImpl(COMPRESSION_CODEC_VALUE, new EmptyMap<>())
        );

        cypress.set(
                Optional.of(transactionId),
                false,
                YPath.simple(YtPathUtil.generatePath(tablePath, "@" + ERASURE_CODEC_KEY)),
                new YTreeStringNodeImpl(ERASURE_CODEC_VALUE, new EmptyMap<>())
        );
    }

    public void shutdown() {
        isShutdown = true;
        logger.info("Received shutdown call");
    }
}
