package ru.yandex.direct.jobs.directdb.service;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Suppliers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyData;
import ru.yandex.direct.env.EnvironmentType;

import static ru.yandex.direct.common.db.PpcPropertyNames.HOME_DIRECT_DB_FULL_WORK;

@Service
@ParametersAreNonnullByDefault
public class HomeDirectDbFullWorkPropObtainerService {
    private static final Logger logger = LoggerFactory.getLogger(HomeDirectDbFullWorkPropObtainerService.class);

    // 5 часов обычно хватает, чтобы все джобы завершились и в таблицу operations проставился статус DONE.
    // Если не хватило, то лучше продлить руками, заново выставив флаг.
    private static final int FULL_WORK_PROPERTY_TTL_HOURS = 5;

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final EnvironmentType environmentType;
    private final PpcProperty<Boolean> fullWorkPpcProperty;
    private final Supplier<LocalDateTime> fullWorkLastChangeCache;

    public HomeDirectDbFullWorkPropObtainerService(PpcPropertiesSupport ppcPropertiesSupport,
                                                   EnvironmentType environmentType
    ) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.environmentType = environmentType;
        this.fullWorkPpcProperty = ppcPropertiesSupport.get(HOME_DIRECT_DB_FULL_WORK, Duration.ofSeconds(30));
        this.fullWorkLastChangeCache = Suppliers.memoizeWithExpiration(this::getLastChange, 30, TimeUnit.SECONDS);
    }

    private LocalDateTime getLastChange() {
        PpcPropertyData<String> fullWorkData = ppcPropertiesSupport
                .getFullByNames(Set.of(HOME_DIRECT_DB_FULL_WORK.getName()))
                .get(HOME_DIRECT_DB_FULL_WORK.getName());
        if (fullWorkData == null) {
            return null;
        }
        return fullWorkData.getLastChange();
    }

    public boolean isFullWorkEnabled() {
        // на проде флаг игнорируется и всегда возвращается true, дабы никто случайно не остановил выгрузку
        // локально тоже игнорируется, дабы не мешать тестовым инстансам
        if (environmentType == EnvironmentType.PRODUCTION || environmentType == EnvironmentType.DEVELOPMENT) {
            return true;
        }
        boolean fullWorkEnabled = fullWorkPpcProperty.getOrDefault(false);

        // проверяем TTL
        if (fullWorkEnabled) {
            LocalDateTime lastChange = fullWorkLastChangeCache.get();

            if (lastChange == null) {
                fullWorkEnabled = false;
            } else if (lastChange.plusHours(FULL_WORK_PROPERTY_TTL_HOURS).isBefore(LocalDateTime.now())) {
                logger.info("{} ppc property value has expired (last change was at {}), setting to false",
                        HOME_DIRECT_DB_FULL_WORK.getName(), lastChange);
                fullWorkPpcProperty.set(false);
                fullWorkEnabled = false;
            }
        }
        return fullWorkEnabled;
    }
}
