package ru.yandex.direct.jobs.directdb.service;

import java.time.LocalDate;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.time.temporal.WeekFields;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.exceptions.OperationRunningException;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.request.CreateNode;
import ru.yandex.yt.ytclient.proxy.request.LockMode;
import ru.yandex.yt.ytclient.proxy.request.ObjectType;

import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;

/**
 * Прореживание архива //home/direct/db-archive
 * <p>
 * - За неделю храним все,
 * - За 10 недель храним архив в неделю (любой)
 * - Более - оставляем только один архив в месяц
 */
@Service
@ParametersAreNonnullByDefault
public class HomeDirectDbResamplingService {

    private static final Logger logger = LoggerFactory.getLogger(HomeDirectDbResamplingService.class);

    private static final String HOME_DB_PATH = "db-archive";
    private static final int RETAIN_PERIOD_DAILY = 7;
    private static final int RETAIN_PERIOD_WEEKLY = 7 * 10;
    private static final int RETAIN_PERIOD_MONTHLY = 100 * 12 * 31;
    private static final String RESAMPLING_LOCK_NODE_PATH = "resampling_lock";

    private final YtProvider ytProvider;
    private final HomeDirectDbArchivingService archivingService;

    public HomeDirectDbResamplingService(YtProvider ytProvider, HomeDirectDbArchivingService archivingService) {
        this.ytProvider = ytProvider;
        this.archivingService = archivingService;
    }

    public void resample(LocalDate now, YtCluster ytCluster) {
        String home = ytProvider.getClusterConfig(ytCluster).getHome();
        String path = YtPathUtil.generatePath(home, relativePart(), HOME_DB_PATH);

        logger.info("Getting folders from {}", path);
        Cypress cypress = ytProvider.get(ytCluster).cypress();
        YTreeNode folders = cypress.get(YPath.simple(path));

        Set<LocalDate> weeksKept = new HashSet<>();
        Set<LocalDate> monthsKept = new HashSet<>();

        YtDynamicOperator ytDynamicOperator = ytProvider.getDynamicOperator(ytCluster);
        ApiServiceTransactionOptions transactionOptions = new ApiServiceTransactionOptions(ETransactionType.TT_MASTER)
                .setSticky(true)
                .setPing(true);

        try {
            ytDynamicOperator.runInTransaction(tx -> {
                takeLock(tx, path, ytCluster);

                folders
                        .asMap()
                        .keySet()
                        .stream()
                        .sorted()
                        .filter(folder -> shouldRemove(folder, now, monthsKept, weeksKept))
                        .forEach(folder -> removePath(home, cypress, folder));

                archivingService.archive(ytCluster, now);
            }, transactionOptions);
        } catch (OperationRunningException e) {
            logger.warn("Can't get lock for resampling", e);
        }
    }

    /**
     * Джоба может выполняться неопределенное время (например, около 3 часов), а запускаться чаще,
     * поэтому берем лок против одновременного архивирования таблиц. Лок автоматически снимается, когда
     * завершается транзакция.
     */
    private void takeLock(ApiServiceTransaction transaction, String homeDbPath, YtCluster ytCluster) {
        String lockPath = YtPathUtil.generatePath(homeDbPath, RESAMPLING_LOCK_NODE_PATH);

        boolean lockNodeExists = transaction.existsNode(lockPath).join(); // IGNORE-BAD-JOIN DIRECT-149116
        // Создаём lock-ноду, если ее нет.
        if (!lockNodeExists) {
            logger.warn("Lock node {} was not found on cluster {}, node will be created", lockPath, ytCluster);
            transaction.createNode(
                    new CreateNode(lockPath, ObjectType.BooleanNode)
                            .setRecursive(true)
                            .setIgnoreExisting(false)
            ).join(); // IGNORE-BAD-JOIN DIRECT-149116
            logger.info("Lock node {} is created on cluster {}", lockPath, ytCluster);
        }

        // Берем лок. Если не удалось, то выкинется исключение.
        logger.info("Trying to take lock on node {} on cluster {}", lockPath, ytCluster);
        transaction.lockNode(lockPath, LockMode.Exclusive).join(); // IGNORE-BAD-JOIN DIRECT-149116
        logger.info("Lock was successfully taken on node {} on cluster {}", lockPath, ytCluster);
    }

    private void removePath(String home, Cypress cypress, String folder) {
        String path = YtPathUtil.generatePath(home, relativePart(), HOME_DB_PATH, folder);
        logger.info("Removing {}", path);
        cypress.remove(YPath.simple(path));
    }

    private boolean shouldRemove(String folder, LocalDate now, Set<LocalDate> monthsKept, Set<LocalDate> weeksKept) {
        logger.info("Checking should we remove {}", folder);
        try {
            LocalDate folderDate = LocalDate.parse(folder);
            long numberOfDaysUntilNow = folderDate.until(now, ChronoUnit.DAYS);

            if (numberOfDaysUntilNow <= RETAIN_PERIOD_DAILY) {
                logger.info(
                        "{} should not be removed because daily retain period not finished yet ({})",
                        folder,
                        RETAIN_PERIOD_DAILY
                );
                return false;
            }

            logger.info("Daily retain period ({} days) already finished for {}", RETAIN_PERIOD_DAILY, folder);
            logger.info("Checking should we delete {} by weekly ({} days) retain period", folder, RETAIN_PERIOD_WEEKLY);

            if (numberOfDaysUntilNow <= RETAIN_PERIOD_WEEKLY) {
                logger.info("Determining start of week for {}", folder);
                LocalDate startOfWeek = folderDate.with(WeekFields.of(Locale.GERMANY).dayOfWeek(), 1L);
                logger.info("Start of week for {} is {}", folder, startOfWeek);

                logger.info("We have to save at least one folder from week");
                if (!weeksKept.contains(startOfWeek)) {
                    logger.info("There are no saved folders from this week, so {} we have to keep", folder);
                    weeksKept.add(startOfWeek);
                    return false;
                }
            }
            if (numberOfDaysUntilNow <= RETAIN_PERIOD_MONTHLY) {
                logger.info("Checking should we delete {} by monthly retain period", folder);

                logger.info("Determining start of month for {}", folder);
                LocalDate startOfMonth = folderDate.withDayOfMonth(1);
                logger.info("Start of month for {} is {}", folder, startOfMonth);

                logger.info("We have to save at least one folder from month");
                if (!monthsKept.contains(startOfMonth)) {
                    logger.info("There are no saved folders from this month, so {} we have to keep", folder);
                    monthsKept.add(startOfMonth);
                    return false;
                }
            }
            logger.info("{} should be removed", folder);
            return true;
        } catch (DateTimeParseException e) {
            return false;
        }
    }

    public void shutdown() {
        archivingService.shutdown();
    }
}
