package ru.yandex.direct.jobs.directdb.service;

import java.time.Duration;
import java.util.Collection;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.jobs.directdb.model.SnapshotAttributes;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeEntityNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static ru.yandex.direct.common.db.PpcPropertyNames.YT_PROD_MYSQL_SHARDS_COUNT;

/**
 * Отдает список пар атрибутов [finished, operations] для каждого элемента в узле
 */
@Service
@ParametersAreNonnullByDefault
public class SnapshotUserAttributeService {
    private static final Logger logger = LoggerFactory.getLogger(SnapshotUserAttributeService.class);

    private static final String FINISHED_ATTRIBUTE = "snapshot-finished";
    private static final String OPERATIONS_ATTRIBUTE = "operations";
    private static final Set<String> ATTRIBUTES = Set.of(FINISHED_ATTRIBUTE, OPERATIONS_ATTRIBUTE);
    @SuppressWarnings("squid:S1075")
    private static final String SOURCE_SNAPSHOT_PATH = "//home/direct/mysql-sync";
    private final YtProvider ytProvider;
    private final ShardHelper shardHelper;
    private final PpcProperty<Integer> ytProdMysqlShardsCountProperty;

    public SnapshotUserAttributeService(YtProvider ytProvider, ShardHelper shardHelper,
                                        PpcPropertiesSupport ppcPropertiesSupport) {
        this.ytProvider = ytProvider;
        this.shardHelper = shardHelper;
        this.ytProdMysqlShardsCountProperty = ppcPropertiesSupport.get(YT_PROD_MYSQL_SHARDS_COUNT, Duration.ofMinutes(5));
    }

    public Collection<SnapshotAttributes> getConvertedUserAttributes(YtCluster cluster) {
        String path = YtPathUtil.generatePath(SOURCE_SNAPSHOT_PATH);

        logger.info("Getting listing with attributes from {}", path);
        YPath cypressPath = YPath.simple(path);
        Cypress cypress = ytProvider.get(cluster).cypress();
        YTreeNode folders = cypress.get(cypressPath, ATTRIBUTES);

        return folders
                .asMap()
                .entrySet()
                .stream()
                .map(
                        tuple -> Pair.of(
                                tuple.getKey(),
                                cypress.get(
                                        YPath.simple(YtPathUtil.generatePath(path, tuple.getKey())),
                                        ATTRIBUTES
                                )
                        )
                )
                .map(tuple -> convertTupleToSnapshotAttributes(path, tuple))
                .collect(Collectors.toList());
    }

    private SnapshotAttributes convertTupleToSnapshotAttributes(String path, Pair<String, YTreeNode> tuple) {
        String folderName = tuple.getKey();

        boolean finished = false;
        YTreeNode value = tuple.getValue();
        if (!(value instanceof YTreeEntityNode)) {
            finished = isFinished(value.mapNode());
        }

        return new SnapshotAttributes(YtPathUtil.generatePath(path, folderName), finished);
    }

    private boolean isFinished(YTreeMapNode ppcFolders) {
        var shardsCount = shardHelper.dbShards().size();
        if (!Environment.getCached().isProduction()) {
            var ytProdMysqlShardsCount = ytProdMysqlShardsCountProperty.getOrDefault(0);
            if (ytProdMysqlShardsCount > 0) {
                shardsCount = ytProdMysqlShardsCount;
            }
        }
        for (int i = 1; i <= shardsCount; i++) {
            Optional<YTreeNode> maybeFinished = ppcFolders
                    .get("ppc:" + i)
                    .flatMap(yTreeNode -> yTreeNode.getAttribute(FINISHED_ATTRIBUTE));

            if (maybeFinished.isEmpty()) {
                return false;
            }

            if (!maybeFinished.get().boolValue()) {
                return false;
            }
        }
        return true;
    }

}
