package ru.yandex.direct.jobs.directdb.service;

import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.currency.model.CurrencyRate;
import ru.yandex.direct.core.entity.currency.repository.CurrencyRateRepository;
import ru.yandex.direct.currency.CurrencyCode;

import static ru.yandex.direct.common.db.PpcPropertyNames.HOME_DIRECT_DB_POOL;

@Service
@ParametersAreNonnullByDefault
public class YqlClasspathObtainerService {
    private static final Logger logger = LoggerFactory.getLogger(YqlClasspathObtainerService.class);

    private final ResourcePatternResolver resourcePatternResolver;
    private final ResourceLoader resourceLoader;
    private final CurrencyRateRepository currencyRateRepository;
    private final PpcProperty<String> poolPpcProp;

    public YqlClasspathObtainerService(
            ResourcePatternResolver resourcePatternResolver,
            ResourceLoader resourceLoader,
            PpcPropertiesSupport ppcPropertiesSupport,
            CurrencyRateRepository currencyRateRepository
    ) {
        this.resourcePatternResolver = resourcePatternResolver;
        this.resourceLoader = resourceLoader;
        this.currencyRateRepository = currencyRateRepository;
        this.poolPpcProp = ppcPropertiesSupport.get(HOME_DIRECT_DB_POOL, Duration.ofMinutes(1));
    }

    /**
     * Возвращает YQL-запросы с названием из classpath:export/home-direct-db. Ко всем запросам в начало запроса
     * добавляется контент файла classpath:export/home-direct-db/include/common.yql.
     *
     * @return Коллекция пар название запроса -> сам YQL запрос.
     */
    public Collection<Pair<String, String>> obtainYqlQueriesFromClassPath() {
        try {
            return getPairs();
        } catch (IOException e) {
            logger.warn("There is an issue with obtaining query list", e);
            return Collections.emptyList();
        }
    }

    private Resource[] getYqlResources() throws IOException {
        return resourcePatternResolver.getResources("classpath:export/home-direct-db/*.yql");
    }

    private Collection<Pair<String, String>> getPairs() throws IOException {
        String commonYql = renderCommonYql();
        Resource[] resources = getYqlResources();
        return Arrays
                .stream(resources)
                .map(resource -> convertResourceToPair(resource, commonYql))
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.toList());
    }

    /**
     * @return список названий запросов (без ".yql")
     */
    public List<String> getYqlQueryNames() {
        try {
            return Arrays.stream(getYqlResources())
                    .map(Resource::getFilename)
                    .map(filename -> filename.substring(0, filename.length() - 4))
                    .collect(Collectors.toList());
        } catch (IOException e) {
            logger.error("There is an issue with obtaining query list", e);
            return Collections.emptyList();
        }
    }

    /**
     * Рендерит общую часть для всех YQL запросов-выгрузок, в которой выставляется пул,
     * регистрируются полезные функции и выставляются прочие настройки.
     * <p>
     * См. файл export/home-direct-db/include/common.yql в ресурсах.
     * <p>
     * Функция $convertToRub генерируется на лету
     */
    private String renderCommonYql() throws IOException {
        Resource includeResource = resourceLoader.getResource("classpath:export/home-direct-db/include/common.yql");
        try (InputStream inputStream = includeResource.getInputStream()) {
            String include = new String(inputStream.readAllBytes());
            return include + renderCurrencyConverter();
        }
    }

    /**
     * Генерирует YQL код функций конвертации валюты
     * <p>
     * {@code $convertToRub(<сумма в валюте>, <валюта>)}
     * <p>
     * Для конвертации берутся самые свежие курсы из {@code ppcdict.currency_rates}
     */
    private String renderCurrencyConverter() {
        List<CurrencyRate> lastCurrencyRates = currencyRateRepository.getLastCurrencyRates(LocalDate.now());

        // функция получения курса валюты к рублю
        StringBuilder currencyConverter = new StringBuilder(
                "\n" +
                "$getCurrencyRate = ($code) -> {\n" +
                "    RETURN\n" +
                "        CASE $code\n" +
                "            WHEN 'RUB' THEN 1.0\n" +
                "            WHEN 'YND_FIXED' THEN 30.0\n");

        lastCurrencyRates.stream()
                .filter(currencyRate -> currencyRate.getCurrencyCode() != CurrencyCode.YND_FIXED &&
                        currencyRate.getCurrencyCode() != CurrencyCode.RUB)
                .forEach(currencyRate -> currencyConverter.append(String.format(
                        Locale.US, // чтобы была точка, а не запятая
                        "            WHEN '%s' THEN %.6f -- as of %s%n",
                        currencyRate.getCurrencyCode(), currencyRate.getRate(), currencyRate.getDate())));

        currencyConverter.append(
                "            ELSE NULL\n" +
                "        END\n" +
                "};\n");

        // функция конвертации суммы из одной валюты в рубли по переданному курсу
        currencyConverter.append(
                "$countCurrencySum = ($sum, $rate) -> {\n" +
                "    RETURN IF($rate IS NOT NULL, Math::Round(CAST($sum AS Double) * CAST($rate AS Double), -6), " +
                "NULL);\n" +
                "};\n");

        // функция конвертации суммы из одной валюты в рубли по ее названию
        currencyConverter.append(
                "$convertToRub = ($sum, $code) -> {\n" +
                "    RETURN CAST($countCurrencySum($sum, $getCurrencyRate($code)) AS String);\n" +
                "};\n");

        return currencyConverter.toString();
    }

    private Optional<Pair<String, String>> convertResourceToPair(Resource resource, String commonYql) {
        try {
            return convertResourceToYqlWithName(resource, commonYql);
        } catch (IOException e) {
            logger.warn(String.format("No such file in classpath %s", resource.getFilename()), e);
        }
        return Optional.empty();
    }

    private Optional<Pair<String, String>> convertResourceToYqlWithName(Resource resource,
                                                                        String commonYql) throws IOException {
        try (InputStream contentInputStream = resource.getInputStream()) {
            String ytPool = Optional
                    .ofNullable(poolPpcProp.get())
                    .orElse("home-direct-db-testing");
            String content = new String(contentInputStream.readAllBytes());
            return Optional.of(
                    Pair.of(
                            resource.getFilename(),
                            "\nPRAGMA yt.Pool = '" + ytPool + "';\n" + commonYql + content
                    )
            );
        }
    }
}
