package ru.yandex.direct.jobs.export;

import java.time.LocalDate;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Rate;
import ru.yandex.direct.currency.RateUtils;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.jobs.util.yql.CommonYqlExport;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;

/**
 * Создание в YT таблички со всеми кампаниями Директа (Бродматч будет забирать их прямо оттуда)
 * <p>
 * Экспорт групп и баннеров пока остался на Perl:
 * https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/bmAllBanners.pl
 */
@Hourglass(periodInSeconds = 3600, needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 4),
        tags = {DIRECT_PRIORITY_1_NOT_READY, CheckTag.YT, CheckTag.GROUP_INTERNAL_SYSTEMS},
        needCheck = ProductionOnly.class)
@ParameterizedBy(parametersSource = DirectExportYtClustersParametersSource.class)
@ParametersAreNonnullByDefault
public class CampaignsForBroadmatchExporter extends DirectParameterizedJob<YtCluster> {
    private static final Logger logger = LoggerFactory.getLogger(CampaignsForBroadmatchExporter.class);

    private static final String QUERY_RESOURCE = "classpath:///export/bmCampaigns.yql";
    private static final String EXPORT_PATH = "export/bm/bm_campaigns";

    /**
     * За сколько последних дней учитываем статистику
     */
    private static final int STAT_DAYS = 30;

    /**
     * Курс нашей y.е. в рублях (используем значение с НДС).
     * Можно было захардкодить 30, но так лучше "связность" кода.
     */
    private static final Rate CONV_UNIT_RATE = RateUtils.get(CurrencyCode.RUB);

    private final CommonYqlExport.Builder exportBuilder;
    private final DirectExportYtClustersParametersSource parametersSource;

    @Autowired
    public CampaignsForBroadmatchExporter(DirectExportYtClustersParametersSource parametersSource,
                                          YtProvider ytProvider) {
        this.parametersSource = parametersSource;

        exportBuilder = new CommonYqlExport.Builder(logger, ytProvider, QUERY_RESOURCE, EXPORT_PATH)
                .withBindingsProvider(CampaignsForBroadmatchExporter::getBindings);
    }

    @Override
    public void execute() {
        YtCluster ytCluster = parametersSource.convertStringToParam(getParam());
        exportBuilder.build(ytCluster).generateIfNeeded();
    }

    private static Object[] getBindings(CommonYqlExport.Context ctx) {
        long updateTimeFrom = LocalDate.now().minusDays(STAT_DAYS).atStartOfDay(MSK).toEpochSecond();

        return new Object[]{
                ctx.getDestinationTablePathAsString(),
                updateTimeFrom,
                CONV_UNIT_RATE.getWithVAT()
        };
    }
}
