package ru.yandex.direct.jobs.export;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;

import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRODUCT_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.juggler.check.model.CheckTag.YT;

/**
 * Джоба для создания/обнвления справочника по ClusterDomain.
 * Запускается несколько раз в день выполняя YQL запрос.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 9),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1, YT, DIRECT_PRODUCT_TEAM})
@Hourglass(cronExpression = "0 0 0/3 * * ?", needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 1, hours = 1, minutes = 30),
        needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_PRIORITY_1, YT, JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "0 0 2/12 * * ?", needSchedule = NonProductionEnvironment.class)
@ParameterizedBy(parametersSource = DirectExportYtClustersParametersSource.class)
@ParametersAreNonnullByDefault
public class ClusterDomainExporter extends DirectParameterizedJob<YtCluster> {

    // путь к таблице на кластере куда будет выкладывать таблица справочника
    private static final String OUTPUT_TABLE_PATH = "/dict/ClusterDomain";
    private static final Logger logger = LoggerFactory.getLogger(ClusterDomainExporter.class);

    // По умолчанию джоба выключена, включается через PPC_PROPERTIES.EXPORT_CLUSTER_DOMAIN_ENABLED
    private static final String DEFAULT_EXPORT_CLUSTER_DOMAIN_ENABLED = "0";

    private final YtProvider ytProvider;

    private DirectExportYtClustersParametersSource parametersSource;

    private final PpcPropertiesSupport ppcPropertiesSupport;

    private static final String CLUSTER_DOMAIN_YQL_QUERY =
            String.join("\n", new ClassPathResourceInputStreamSource("export/ClusterDomain.sql")
                    .readLines());

    @Autowired
    public ClusterDomainExporter(
            YtProvider ytProvider,
            PpcPropertiesSupport ppcPropertiesSupport,
            DirectExportYtClustersParametersSource parametersSource) {
        this.ytProvider = ytProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.parametersSource = parametersSource;
    }

    private boolean isEnabled() {
        String jobEnabledProperty = ppcPropertiesSupport
                .find(PpcPropertyEnum.EXPORT_CLUSTER_DOMAIN_ENABLED.getName())
                .filter(StringUtils::isNumeric)
                .map(String::valueOf)
                .orElse(DEFAULT_EXPORT_CLUSTER_DOMAIN_ENABLED);

        logger.info("Loaded ppc_property {} = {}", PpcPropertyEnum.EXPORT_CLUSTER_DOMAIN_ENABLED.getName(), jobEnabledProperty);
        return "1".equals(jobEnabledProperty);
    }

    @Override
    public void execute() {

        if (!isEnabled()) {
            return;
        }

        YtCluster ytCluster = parametersSource.convertStringToParam(getParam());
        YtClusterConfig ytClusterConfig = ytProvider.getClusterConfig(ytCluster);
        YtOperator ytOperator = ytProvider.getOperator(ytCluster, YtSQLSyntaxVersion.SQLv1);

        final String homePath = ytClusterConfig.getHome();
        final String tablePath = YtPathUtil.generatePath(homePath, relativePart(), OUTPUT_TABLE_PATH);

        ytOperator.yqlExecute(CLUSTER_DOMAIN_YQL_QUERY, tablePath);
    }
}
