package ru.yandex.direct.jobs.export;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.jobs.util.yql.CommonYqlExport;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;

/**
 * Создание в YT таблички со всеми баннерами Директа из активных кампаний для Фокусировщика
 */
@Hourglass(periodInSeconds = 3600, needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 4),

        //PRIORITY: уточнить
        tags = {DIRECT_PRIORITY_1_NOT_READY, CheckTag.YT, CheckTag.GROUP_INTERNAL_SYSTEMS},
        needCheck = ProductionOnly.class)
@ParameterizedBy(parametersSource = DirectExportYtClustersParametersSource.class)
@ParametersAreNonnullByDefault
public class YaContextAllBannersExporter extends DirectParameterizedJob<YtCluster> {
    private static final Logger logger = LoggerFactory.getLogger(YaContextAllBannersExporter.class);

    private static final String QUERY_RESOURCE = "classpath:///export/yaContextAllBanners.yql";
    private static final String EXPORT_PATH = "export/yacontext/yacontext_banners";
    private static final int LAST_SHOW_TIME_DAYS_GAP = 7;

    private final CommonYqlExport.Builder exportBuilder;
    private final DirectExportYtClustersParametersSource parametersSource;

    @Autowired
    public YaContextAllBannersExporter(YtProvider ytProvider,
                                       DirectExportYtClustersParametersSource parametersSource) {
        this.parametersSource = parametersSource;

        exportBuilder = new CommonYqlExport.Builder(logger, ytProvider, QUERY_RESOURCE, EXPORT_PATH)
                .withBindingsProvider(YaContextAllBannersExporter::getBindings);
    }

    @Override
    public void execute() {
        YtCluster ytCluster = parametersSource.convertStringToParam(getParam());
        exportBuilder.build(ytCluster).generateIfNeeded();
    }

    private static Object[] getBindings(CommonYqlExport.Context ctx) {
        LocalDate borderDate = ctx.getDbUploadDate().minusDays(LAST_SHOW_TIME_DAYS_GAP);
        return new Object[]{
                borderDate.format(DateTimeFormatter.ISO_LOCAL_DATE),
                ctx.getDestinationTablePathAsString()
        };
    }
}
