package ru.yandex.direct.jobs.export.feature;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.core.entity.feature.model.FeatureHistory;
import ru.yandex.direct.core.entity.feature.repository.FeatureHistoryRepository;
import ru.yandex.direct.core.entity.feature.repository.FeatureRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRODUCT_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.YT;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 6), needCheck = NonDevelopmentEnvironment.class,
        tags = {DIRECT_PRIORITY_2, YT, DIRECT_PRODUCT_TEAM})
//запускается только на проде и ТС.
// если будет нужно запускать на других средах, то требуется поправить путь до таблицы в FeaturesYtUploader
//чтобы не пересекались выгрузки например ТС и devtest
@Hourglass(cronExpression = "0 0 0/2 * * ?", needSchedule = NonDevelopmentEnvironment.class)
@ParameterizedBy(parametersSource = DirectExportYtClustersParametersSource.class)
public class ExportFeaturesToYtJob extends DirectParameterizedJob<YtCluster> {

    private static final int CHUNK_SIZE = 100;
    private static final RelaxedWorker relaxedWorker = new RelaxedWorker(2.0);
    private static final int MAX_PERCENT = 100;

    private final YtProvider ytProvider;
    private final DirectExportYtClustersParametersSource parametersSource;
    private final FeatureRepository featureRepository;
    private final FeatureHistoryRepository featureHistoryRepository;
    private final UserService userService;

    @Autowired
    public ExportFeaturesToYtJob(
            YtProvider ytProvider,
            DirectExportYtClustersParametersSource parametersSource,
            FeatureRepository featureRepository,
            FeatureHistoryRepository featureHistoryRepository,
            UserService userService) {
        this.ytProvider = ytProvider;
        this.parametersSource = parametersSource;
        this.featureRepository = featureRepository;
        this.featureHistoryRepository = featureHistoryRepository;
        this.userService = userService;
    }

    @Override
    public void execute() {
        YtCluster ytCluster = parametersSource.convertStringToParam(getParam());

        FeaturesYtUploader uploader = new FeaturesYtUploader(ytProvider, ytCluster);
        uploader.createTableIfAbsent();

        var allFeatures = featureRepository.get();
        for (var featureChunk : Iterables.partition(allFeatures, CHUNK_SIZE)) {
            var featureIds = mapList(featureChunk, Feature::getId);
            var featureIdToLogin = getFeatureIdToLogin(featureIds);
            relaxedWorker.runAndRelax(() -> {
                var features = mapList(featureChunk, f -> convertRow(f, featureIdToLogin));
                uploader.upload(features);
            });
        }
    }

    // идентификатор фичи -> логин пользователя, который выставил фичу в 100%
    private Map<Long, String> getFeatureIdToLogin(Collection<Long> featureIds) {
        var featureIdToHistory = featureHistoryRepository.getFeatureIdToHistory(featureIds);
        var featureIdToOperatorUid = getFeatureIdToOperatorUid(featureIdToHistory);

        if (!featureIdToOperatorUid.isEmpty()) {
            var userIdToLogin = StreamEx.of(userService.massGetUser(featureIdToOperatorUid.values()))
                    .mapToEntry(User::getUid, ExportFeaturesToYtJob::getStaffLogin)
                    .toMap();

            return EntryStream.of(featureIdToOperatorUid)
                    .mapValues(userIdToLogin::get)
                    .nonNullValues()
                    .toMap();
        } else {
            return Collections.emptyMap();
        }
    }

    private Map<Long, Long> getFeatureIdToOperatorUid(Map<Long, List<FeatureHistory>> featureIdToHistory) {
        var featureIdToOperatorUid = new HashMap<Long, Long>();
        for (var entry : featureIdToHistory.entrySet()) {
            var featureId = entry.getKey();
            var historyList = entry.getValue();
            for (var history : historyList) {
                var percent = history.getNewSettings().getPercent();
                if (percent == null) {
                    continue;
                }
                if (percent == MAX_PERCENT && history.getOperatorUid() != null) {
                    featureIdToOperatorUid.put(featureId, history.getOperatorUid());
                }
                break;
            }
        }
        return featureIdToOperatorUid;
    }

    private static String getStaffLogin(User user) {
        var result = user.getEmail().split("@");
        if (result.length > 1) {
            return result[0];
        }
        return null;
    }

    private static YtFeature convertRow(Feature feature, Map<Long, String> featureIdToLogin) {
        return new YtFeature()
                .withFeatureId(feature.getId())
                .withFeatureTextId(feature.getFeatureTextId())
                .withFeaturePublicName(feature.getFeaturePublicName())
                .withPercent(feature.getSettings().getPercent())
                .withLogin(featureIdToLogin.get(feature.getId()))
                .withOwner(feature.getSettings().getOriginalOwner())
                .withTicketId(feature.getFeatureTicket());
    }
}
