package ru.yandex.direct.jobs.export.feature;

import java.util.Arrays;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

public class FeaturesYtUploader {
    private static final String FEATURES_EXPORT_PATH = "export/chassis/features";

    private static final String FEATURE_ID_COLUMN = "feature_id";
    private static final String FEATURE_TEXT_ID_COLUMN = "feature_text_id";
    private static final String FEATURES_PUBLIC_NAME_COLUMN = "feature_public_name";
    private static final String PERCENT_COLUMN = "percent";
    private static final String LOGIN_TO_SET_PERCENT = "login";
    private static final String OWNER_COLUMN = "owner";
    private static final String TICKET_ID_COLUMN = "ticket_id";

    private static final Logger logger = LoggerFactory.getLogger(FeaturesYtUploader.class);

    private static final TableSchema schema = new TableSchema.Builder()
            .addKey(FEATURE_ID_COLUMN, ColumnValueType.INT64)
            .addValue(FEATURE_TEXT_ID_COLUMN, ColumnValueType.STRING)
            .addValue(FEATURES_PUBLIC_NAME_COLUMN, ColumnValueType.STRING)
            .addValue(PERCENT_COLUMN, ColumnValueType.INT64)
            .addValue(LOGIN_TO_SET_PERCENT, ColumnValueType.STRING)
            .addValue(OWNER_COLUMN, ColumnValueType.STRING)
            .addValue(TICKET_ID_COLUMN, ColumnValueType.STRING)
            .build();

    private final YtProvider ytProvider;
    private final YtCluster cluster;
    private final String path;

    FeaturesYtUploader(YtProvider ytProvider, YtCluster cluster) {
        this.ytProvider = ytProvider;
        this.cluster = cluster;
        YtClusterConfig ytClusterConfig = ytProvider.getClusterConfig(cluster);
        this.path = YtPathUtil.generatePath(ytClusterConfig.getHome(), FEATURES_EXPORT_PATH);
    }

    void createTableIfAbsent() {
        YtOperator ytOperator = ytProvider.getOperator(cluster);
        Yt yt = ytOperator.getYt();
        YPath table = YPath.simple(path);
        logger.info("cluster: {}, table: {}", cluster, table);

        if (yt.cypress().exists(table)) {
            ytOperator.unmount(table, 5 * 60 * 1000);
            yt.cypress().remove(table);
            logger.info("schema changes, unmounting and start creation");
        }

        try {
            // создаем динамическую таблицу
            logger.info("Creating table");
            yt.cypress().create(table, CypressNodeType.TABLE, true, false,
                    Cf.map("schema", schema.toYTree(), "dynamic", YTree.booleanNode(true)));

            // монтируем таблицу
            logger.info("Mounting table ..");
            ytOperator.mount(table, 60_000);
            logger.info("Successfully mounted table");

        } catch (RuntimeException e) {
            // при ошибке удаляем созданную таблицу
            try {
                if (yt.cypress().exists(table)) {
                    yt.cypress().remove(table);
                }
            } catch (RuntimeException e1) {
                logger.error("Error in cleanup", e1);
            }
            throw e;
        }
    }

    void upload(List<YtFeature> features) {
        YtDynamicOperator operator = ytProvider.getDynamicOperator(cluster);

        logger.info("Updating features");
        operator.runInTransaction(transaction -> replaceWithNewData(transaction, features));
        logger.info("Successfully finished updating");
    }

    private void replaceWithNewData(ApiServiceTransaction transaction, List<YtFeature> features) {
        ModifyRowsRequest insertRequest = new ModifyRowsRequest(path, schema);

        features.forEach(feature -> insertRequest.addInsert(Arrays.asList(
                feature.getFeatureId(),
                feature.getFeatureTextId(),
                feature.getFeaturePublicName(),
                feature.getPercent(),
                feature.getLogin(),
                feature.getOwner(),
                feature.getTicketId())));

        logger.info("Inserting rows");
        transaction.modifyRows(insertRequest).join(); // IGNORE-BAD-JOIN DIRECT-149116
    }
}
