package ru.yandex.direct.jobs.export.feature.diff

import org.joda.time.DateTime
import org.joda.time.Days
import org.slf4j.LoggerFactory
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.jobs.featureschanges.FeaturesLogStartrekService
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion
import ru.yandex.startrek.client.model.Issue
import ru.yandex.startrek.client.model.UserRef

private const val TS_FEATURE_EXPIRED_TAG = "ts_feature_expired"

private const val FIRST_REMIND_IN_DAYS = 7
private const val REMIND_EVERY_DAYS = 3

@JugglerCheck(
    ttl = JugglerCheck.Duration(days = 2),
    tags = [CheckTag.DIRECT_PRIORITY_2], needCheck = ProductionOnly::class,
)
// Функционал заменён джобой FeatureDevelopmentSyncJob, в будущем будет удалена
// @Hourglass(cronExpression = "0 0 1 * * ?", needSchedule = ProductionOnly::class)
class FeatureDiffJob(
    private val startrekService: FeaturesLogStartrekService,
    private val ytProvider: YtProvider,
) : DirectJob() {

    private val logger = LoggerFactory.getLogger(FeatureDiffJob::class.java)

    override fun execute() {
        val cluster = Utils.chooseFreshestCluster(ytProvider)
        val ytOperator = ytProvider.getOperator(cluster, YtSQLSyntaxVersion.SQLv1)
        val tsTable = Utils.resolveTsTable(cluster)

        val prodFeatureMap: Map<String, FeatureRow> = Utils.readFeatureTable(ytOperator, Utils.PROD_YT_TABLE)
        val tsFeatureMap: Map<String, FeatureRow> = Utils.readFeatureTable(ytOperator, tsTable)

        prodFeatureMap.forEach { (prodFeatureTextId, prodFeature) ->
            val tsFeature = tsFeatureMap[prodFeatureTextId]
            if (prodFeature.percent == Utils.ONE_HUNDRED_PERCENT && prodFeature.percent != tsFeature?.percent) {
                createTicketOrAddReminder(prodFeatureTextId, prodFeature, tsFeature)
            }
        }
    }

    private fun createTicketOrAddReminder(featureTextId: String, prodFeature: FeatureRow, tsFeature: FeatureRow?) {
        val tickets = startrekService.findTsFeatureTickets(featureTextId);

        if (tickets.size > 1) {
            logger.warn("More than one ticket found {} for feature {}", tickets, prodFeature.textId)
        }

        if (tickets.isEmpty()) {
            val issue = startrekService.createTsTicket(
                diffSummary(prodFeature.textId),
                diffDescription(prodFeature, tsFeature),
                prodFeature
            )

            addCommentToParentTicket(prodFeature, issue.key);
        } else {
            addReminder(tickets[0], prodFeature, tsFeature);
        }
    }

    private fun diffSummary(textId: String): String {
        return "Включить на ТС фичу: $textId"
    }

    private fun diffDescription(prodFeature: FeatureRow, tsFeature: FeatureRow?): String {
        val featureText = getFeaturesStateMessage(tsFeature, prodFeature)
        return "Включить на ТС фичу. Проверить, что все работает ожидаемо и закрыть тикет.\n" +
            "$featureText\n" +
            "Идентификатор фичи: ${prodFeature.textId}; Имя фичи: \"${prodFeature.name}\"\n" +
            "----\n" +
            "Тикет создан джобой FeatureDiffJob из приложения chassis"
                .trimIndent()
    }

    /**
     * Добавляет комментарий в основной тикет с текстом о том, что был создан сабтикет на включение фичи на ТС
     */
    private fun addCommentToParentTicket(prodFeature: FeatureRow, subTicketKey: String) {
        val parentTicketId = prodFeature.ticketId;
        if (parentTicketId == null) {
            logger.warn("Parent ticket for {} not found", prodFeature.textId)
        }

        startrekService.addTicketComment(parentTicketId, "Создан сабтикет на включение фичи на тс: $subTicketKey")
    }

    /**
     * Добавить в тикет комментарий напоминающий о том, что фичу нужно включить на 100% на ТС.
     * Вместе с комметарием добавляется тег ts_feature_expired
     */
    private fun addReminder(ticket: Issue, prodFeature: FeatureRow, tsFeature: FeatureRow?) {
        if (!shouldRemind(ticket)) {
            logger.info("The reminder time for ticket {} didn't come", ticket.key)
            return
        }

        val assignee = ticket.assignee.toArray(UserRef::class.java)
        val featuresStateText = getFeaturesStateMessage(tsFeature, prodFeature)
        ticket.comment(
            "Пожалуйста, не забудьте включить на ТС фичу. " +
                "Затем проверьте, что все работает ожидаемо и закройте тикет.\n$featuresStateText",
            *assignee
        )

        startrekService.addTicketTag(ticket.key, TS_FEATURE_EXPIRED_TAG)
    }

    private fun getFeaturesStateMessage(tsFeature: FeatureRow?, prodFeature: FeatureRow): String {
        return if (tsFeature == null) "ТС: фича отсутствует в базе. Прод: включена на ${prodFeature.percent}"
        else "ТС: Фича включена на ${tsFeature.percent}. Прод: включена на ${prodFeature.percent}"
    }

    /**
     * график напоминаний: 7 дней, затем, каждые 3 дня
     */
    private fun shouldRemind(ticket: Issue): Boolean {
        val daysSinceTicketCreated = Days.daysBetween(ticket.createdAt.toDateTime(), DateTime.now()).days

        return daysSinceTicketCreated == FIRST_REMIND_IN_DAYS ||
            (daysSinceTicketCreated > FIRST_REMIND_IN_DAYS &&
                0 == (daysSinceTicketCreated - FIRST_REMIND_IN_DAYS) % REMIND_EVERY_DAYS)
    }
}
