package ru.yandex.direct.jobs.export.feature.sync

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Qualifier
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.TranslationService
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToPercent
import ru.yandex.direct.core.entity.feature.model.ClientFeature
import ru.yandex.direct.core.entity.feature.model.Feature
import ru.yandex.direct.core.entity.feature.repository.ClientFeaturesRepository
import ru.yandex.direct.core.entity.feature.repository.FeatureRepository
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService
import ru.yandex.direct.env.Environment
import ru.yandex.direct.env.EnvironmentType
import ru.yandex.direct.env.NonProductionEnvironment
import ru.yandex.direct.feature.FeatureHistoryEvent
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.feature.FeatureType
import ru.yandex.direct.intapi.client.IntapiSmartClient
import ru.yandex.direct.integrations.configuration.IntegrationsConfiguration.PRODUCTION_INTAPI_SMART_CLIENT
import ru.yandex.direct.integrations.configuration.IntegrationsConfiguration.TELEGRAM_CLIENT_DIRECT_FEATURE
import ru.yandex.direct.jobs.configuration.JobsEssentialConfiguration.STARTREK_SESSION_ROBOT_DIRECT_FEATURE
import ru.yandex.direct.jobs.export.feature.diff.Utils
import ru.yandex.direct.jobs.featureschanges.FeaturesLogStartrekService
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.result.Result
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.telegram.client.TelegramClient
import ru.yandex.direct.telegram.client.api.ParseMode
import ru.yandex.startrek.client.Session
import ru.yandex.startrek.client.model.Issue
import ru.yandex.startrek.client.model.IssueCreate
import ru.yandex.startrek.client.model.Relationship
import java.time.Duration

@JugglerCheck(
    ttl = JugglerCheck.Duration(hours = 4),
    needCheck = NonProductionEnvironment::class,
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_DARKKEKS],
        method = [NotificationMethod.TELEGRAM],
    )],
)
@Hourglass(cronExpression = "0 */5 4 * * ?", needSchedule = NonProductionEnvironment::class)
class FeatureDevelopmentSyncJob(
    private val featuresRepository: FeatureRepository,
    private val featureManagingService: FeatureManagingService,
    private val clientFeaturesRepository: ClientFeaturesRepository,
    @Qualifier(TELEGRAM_CLIENT_DIRECT_FEATURE)
    private val telegramClient: TelegramClient,
    @Qualifier(PRODUCTION_INTAPI_SMART_CLIENT)
    private val intapiSmartClient: IntapiSmartClient,
    private val translationService: TranslationService,
    @Qualifier(STARTREK_SESSION_ROBOT_DIRECT_FEATURE)
    private val startrekSession: Session,
    private val featuresLogStartrekService: FeaturesLogStartrekService,
    ppcPropertiesSupport: PpcPropertiesSupport,
) : DirectJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(FeatureDevelopmentSyncJob::class.java)

        // id последней записи в features_history на момент первого запуска джобы
        private const val DEFAULT_LAST_EVENT_ID: Long = 4294

        // id канала, в который будут отправляться уведомления о синхронизации фич
        private const val DIRECT_FEATURES_CHAT_ID: Long = -1001778931086

        // тег для тикетов, заводимых джобой
        private const val TESTS_ISSUE_TAG = "feature-fix-tests"

        // если не получилось определить owner-а фичи
        private const val DEFAULT_ASSIGNEE = "robot-direct-feature"
    }

    private val lastEventIdProperty: PpcProperty<Long> =
        ppcPropertiesSupport.get(PpcPropertyNames.FEATURES_SYNC_LAST_EVENT_ID)

    private val ignoredFeaturesProperty: PpcProperty<Set<String>> =
        ppcPropertiesSupport.get(PpcPropertyNames.FEATURES_SYNC_IGNORED_FEATURES, Duration.ofMinutes(1))

    override fun execute() {
        val lastEventId: Long = lastEventIdProperty.get() ?: DEFAULT_LAST_EVENT_ID

        val events: List<FeatureHistoryEvent> = intapiSmartClient
            .getFeaturesHistory(lastEventId)
            .sortedBy { it.eventId }
            .filter { it.featureTextId != null }
        logger.info("All events: $events")

        events.forEach { event ->
            logger.info("Processing event: $event")
            processEvent(event)
        }
    }

    private fun processEvent(event: FeatureHistoryEvent) {
        val existingFeature: Feature? =
            featuresRepository.getByFeatureTextId(event.featureTextId).orElse(null)
        if (!shouldProcess(event, existingFeature)) {
            logger.info("Will not process $event, existingFeature: $existingFeature")
            lastEventIdProperty.set(event.eventId)
            return
        }

        val feature = existingFeature
            ?: addFeature(event)
        updateFeature(event)
        val deletedFeatures = deleteClientFeatures(feature.id)

        val issue: Issue? = try {
            createIssue(event, feature, deletedFeatures, wasAdded = existingFeature == null)
        } catch (e: RuntimeException) {
            logger.error("Failed to create issue", e)
            null
        }

        lastEventIdProperty.set(event.eventId)

        sendNotification(event, feature, deletedFeatures, wasAdded = existingFeature == null, issue)
    }

    private fun shouldProcess(event: FeatureHistoryEvent, feature: Feature?): Boolean =
        event.featureTextId !in ignoredFeaturesProperty.get().orEmpty()
            // в том числе проверяет, что фича есть в коде
            && FeatureName.fromString(event.featureTextId)?.featureType == FeatureType.TEMP
            // фичу включили на 100% или выключили/удалили после включения на 100%
            && (event.percentAfter == Utils.ONE_HUNDRED_PERCENT
            || (event.percentAfter in listOf(null, Utils.ZERO_PERCENT)
            && event.percentBefore == Utils.ONE_HUNDRED_PERCENT))

    private fun addFeature(event: FeatureHistoryEvent): Feature {
        val featureName = FeatureName.fromString(event.featureTextId)!!

        val featureToAdd = Feature()
            .withFeatureTextId(featureName.getName())
            .withFeaturePublicName(translationService.translate(featureName.humanReadableName))

        logger.info("Adding feature ${featureToAdd.featureTextId}")
        val addResult = featureManagingService.addFeaturesWithDefaultSettings(listOf(featureToAdd))
        if (!addResult.isSuccessful) {
            throw IllegalStateException("Failed to add feature: ${addResult.validationResult.flattenErrors()}")
        }

        return addResult.result.first()
    }

    private fun updateFeature(event: FeatureHistoryEvent) {
        val update = FeatureTextIdToPercent()
            .withTextId(event.featureTextId)
            .withPercent(event.percentAfter ?: 0)

        logger.info("Updating feature ${update.textId} to ${update.percent}")
        val updateResult: Result<Collection<Long>> = featureManagingService.updateFeaturePercent(listOf(update))
        if (!updateResult.isSuccessful) {
            throw IllegalStateException("Failed to enable features: ${updateResult.validationResult.flattenErrors()}")
        }
    }

    private fun deleteClientFeatures(featureId: Long): List<ClientFeature> {
        val clientFeatures = clientFeaturesRepository.getClientsWithFeatures(setOf(featureId))
            .values.flatten()

        logger.info("Deleting ${clientFeatures.size} client features: $clientFeatures")
        clientFeaturesRepository.deleteClientFeatures(clientFeatures)

        return clientFeatures
    }

    private fun createIssue(
        event: FeatureHistoryEvent,
        feature: Feature,
        deletedFeatures: List<ClientFeature>,
        wasAdded: Boolean,
    ): Issue? {
        val environment = Environment.getCached()

        // заводим тикеты только для изменения фичей на ТС
        if (environment != EnvironmentType.TESTING) {
            return null
        }

        val featureIssue: Issue? = featuresLogStartrekService.getTicket(event.featureTextId)

        var description = """
            Фича %%${event.featureTextId}%% в окружении %%$environment%% была автоматически изменена после изменения в production:
        """.trimIndent()

        description += if (wasAdded) {
            "\n - Процент: !!(зел)${event.percentAfter}%!! (добавлена в базу)"
        } else if (feature.settings.percent != event.percentAfter) {
            "\n - Процент: !!${feature.settings.percent}!!% → !!(зел)${event.percentAfter}!!%"
        } else {
            "\n - Процент: ${event.percentAfter}% (без изменений)"
        }

        if (deletedFeatures.isNotEmpty()) {
            description += "\n - <{Клиенты, у которых фича была удалена"
            description += "\n%%"
            deletedFeatures.forEach {
                description += "\n${it.clientId.asLong()}"
            }
            description += "\n%%"
            description += "\n}>"
        }

        description += "\n\n"
        description += """
            Что делать с этим тикетом:
            - Актуализировать тесты, которые используют старое значение фичи
            - Закрыть тикет, если таких тестов нет
        """.trimIndent()

        return startrekSession.issues().create(
            IssueCreate.builder()
                .queue("DIRECT")
                .summary("Актуализировать тесты для фичи: ${event.featureTextId}")
                .description(description)
                .apply { if (featureIssue != null) parent(featureIssue) }
                .tags(TESTS_ISSUE_TAG, event.featureTextId)
                .assignee(event.owner ?: DEFAULT_ASSIGNEE)
                .build()
        )
    }

    private fun sendNotification(
        event: FeatureHistoryEvent,
        feature: Feature,
        deletedFeatures: List<ClientFeature>,
        wasAdded: Boolean,
        issue: Issue?,
    ) {
        var text = "<b>${Environment.getCached().name.lowercase()}</b>:"

        text += "\n<code>${event.featureTextId}</code>:"

        text += if (wasAdded) {
            "\n  Процент: ${event.percentAfter}% (добавлена в базу)"
        } else if (feature.settings.percent != event.percentAfter) {
            "\n  Процент: ${feature.settings.percent}% → ${event.percentAfter}%"
        } else {
            "\n  Процент: ${event.percentAfter}% (без изменений)"
        }

        if (deletedFeatures.isNotEmpty()) {
            text += "\n  Клиентов, у которых фича была удалена: ${deletedFeatures.size}"
        }

        if (issue != null) {
            val issueLink = """<a href="https://st.yandex-team.ru/${issue.key}">${issue.key}</a>"""
            text += "\n  Тикет на актуализацию тестов: $issueLink"
        }

        telegramClient.sendMessage(DIRECT_FEATURES_CHAT_ID, text = text, parseMode = ParseMode.HTML)
    }
}

