package ru.yandex.direct.jobs.featureschanges;

import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.util.List;

import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.featureschanges.exception.TicketNotFound;
import ru.yandex.direct.jobs.featureschanges.handlers.FeaturesChangesLogHandler;
import ru.yandex.direct.jobs.featureschanges.handlers.FeaturesChangesLogHandlerFactory;
import ru.yandex.direct.jobs.featureschanges.model.ClientsFeaturesChangesLogData;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.startrek.client.error.StartrekInternalClientError;
import ru.yandex.startrek.client.error.StartrekInternalServerError;
import ru.yandex.startrek.client.error.UnauthorizedException;

import static java.time.LocalDateTime.now;
import static java.util.stream.Collectors.joining;
import static ru.yandex.direct.common.db.PpcPropertyNames.CLIENTS_FEATURES_CHANGES_LOG_JOB_IGNORED_FEATURES_IDS;
import static ru.yandex.direct.common.db.PpcPropertyType.LOCAL_DATE_TIME;
import static ru.yandex.direct.common.db.PpcPropertyType.STRING;
import static ru.yandex.direct.dbutil.wrapper.SimpleDb.CLICKHOUSE_CLOUD;
import static ru.yandex.direct.jobs.featureschanges.FeaturesChangesLogFilterHelper.filterFirstItemsByOneTicket;
import static ru.yandex.direct.jobs.featureschanges.FeaturesChangesLogMappers.CLIENTS_FEATURES_ROW_MAPPER;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Джоба следит за обновлениями в таблице clients_features и записывает изменения в стартрек.
 * Ради упрощения кода один запуск джобы обрабатывает один тикет.
 * <p>
 * Решение сделано на jobs, а не на ess, из-за того решение на jobs позволяет агрегировать события и писать одним
 * комментарием.
 */

@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 30),
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM},
        needCheck = ProductionOnly.class
)
@Hourglass(periodInSeconds = 60 * 5, needSchedule = ProductionOnly.class)
public class ClientsFeaturesChangesLogJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(ClientsFeaturesChangesLogJob.class);
    private static final int ROWS_FETCH_LIMIT = 10000;
    private static final PpcPropertyName<LocalDateTime> LAST_EVENT_TIME_PPC_PROPERTY =
            new PpcPropertyName<>("features_bin_logs_reader.clients_features_last_event_datetime", LOCAL_DATE_TIME);
    private static final String IGNORED_FEATURES_IDS_PLACEHOLDER = "{IGNORED_FEATURES_IDS}";
    private static final long ALLOWED_HOURS_LAG = 2L;

    private static final PpcPropertyName<String> LAST_EVENT_UNIQUE_ID_PPC_PROPERTY =
            new PpcPropertyName<>("features_bin_logs_reader.clients_features_last_event_unique_id", STRING);

    private static final String SQL_TEMPLATE =
            "SELECT datetime, primary_key, operation, row.name, row.value, reqid, method" +
                    "        FROM binlog_rows_v2 as bnlgs " +
                    "       WHERE datetime >= ? " +
                    // события с разных шардов могут приходить не в порядке datetime (приходить с отставанием)
                    "         AND datetime < minus(now(), toIntervalMinute(5)) " +
                    "         AND table='clients_features' " +
                    // сразу исключаем часто включаемые фичи
                    "         AND substringUTF8(primary_key, positionUTF8(primary_key, ',') + 1) " +
                    "             not in (" + IGNORED_FEATURES_IDS_PLACEHOLDER + ") " +
                    "    ORDER BY datetime, primary_key" +
                    "       LIMIT " + ROWS_FETCH_LIMIT;


    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final DatabaseWrapperProvider dbProvider;
    private final FeaturesChangesLogHandlerFactory handlerFactory;
    private final PpcLogCmdUserDataService userDataService;

    @Autowired
    public ClientsFeaturesChangesLogJob(PpcPropertiesSupport ppcPropertiesSupport, DatabaseWrapperProvider dbProvider,
                                        FeaturesChangesLogHandlerFactory handlerFactory,
                                        PpcLogCmdUserDataService userDataService) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.dbProvider = dbProvider;
        this.handlerFactory = handlerFactory;
        this.userDataService = userDataService;
    }

    /**
     * 1) Достаем пачку
     * 2) Отфильтровываем относительно uniqueId и берем подряд идущие события, относящиеся к одной фиче
     * 3) Обрабатываем эти события
     * 4) Обновляем ppc_properties
     * 5) Обновляем локальную lastEventUniqueId
     * 6) Переходим к пункту 2
     */
    @Override
    public void execute() {
        String lastEventUniqueId = getLastEventUniqueId();
        LocalDateTime lastEventDatetime = getLastEventDatetime();
        List<ClientsFeaturesChangesLogData> fullLogDataList = selectRows(lastEventDatetime);

        if (isJobInLoop(fullLogDataList, lastEventUniqueId)) {
            logger.warn("Job in loop. Move last_event_datetime by one second");
            moveLastEventTimeByOneSecond(lastEventDatetime);
            return;
        }

        List<ClientsFeaturesChangesLogData> currentLogDataList =
                filterFirstItemsByOneTicket(fullLogDataList, lastEventUniqueId);

        // Если все события в fullLogDataList уже были обработаны, то сразу завершаем работу джобы,
        // чтобы не проверять checkLag и не заполнять fullLogDataList логинами
        if (currentLogDataList.isEmpty()) {
            return;
        }

        userDataService.enrichWithUserData(fullLogDataList, lastEventDatetime);

        while (!currentLogDataList.isEmpty()) {
            handleLogDataList(currentLogDataList);
            lastEventUniqueId = Iterables.getLast(currentLogDataList).getUniqueId();

            currentLogDataList = filterFirstItemsByOneTicket(fullLogDataList, lastEventUniqueId);
        }

        checkLag(fullLogDataList);
    }

    private void handleLogDataList(List<ClientsFeaturesChangesLogData> logDataList) {
        if (logDataList.isEmpty()) {
            return;
        }

        boolean isBrokenDataList = logDataList.stream().anyMatch(ClientsFeaturesChangesLogData::isBrokenData);
        // после filterFirstItemsByOneTicket либо все события некорректные, либо все корректные.
        // Так как feature_id у корректных и у некорректных не совпадает.
        if (isBrokenDataList) {
            // сразу сдвигаем lastEventDatetime
            updateLastEventDatetimeAndUniqueId(logDataList);
            return;
        }

        try {
            FeaturesChangesLogHandler handler = handlerFactory.createClientsFeaturesHandler(logDataList);
            handler.handle();
        } catch (StartrekInternalClientError | StartrekInternalServerError | UnauthorizedException e) {
            //если стартрек недоступен, то джоба должна обработать эти события еще раз, при следующем запуске.
            logger.info("An error occurred while accessing to the Startrek");
            throw e;
        } catch (TicketNotFound e) {
            // ждем пока тикет будет создан в FeaturesHistoryChangesLogJob
            // кидаем Exception, чтобы сработал алерт в джаглере
            logger.warn("Ticket for {} not found", e.getFeatureTextId());
            throw e;
        } catch (RuntimeException suppressed) {
            // любые другие ошибки подавляем и обновляем LastEventDatetime
            logger.warn("Unable to handle events", suppressed);
        }

        updateLastEventDatetimeAndUniqueId(logDataList);
    }


    private List<ClientsFeaturesChangesLogData> selectRows(LocalDateTime lastEventDatetime) {
        String sql = SQL_TEMPLATE.replace(IGNORED_FEATURES_IDS_PLACEHOLDER, getIgnoredFeaturesIds());

        Object[] bindings = getSqlBindings(lastEventDatetime);
        List<ClientsFeaturesChangesLogData> logDataList = dbProvider.get(CLICKHOUSE_CLOUD)
                .query(sql, bindings, CLIENTS_FEATURES_ROW_MAPPER);

        if (!logDataList.isEmpty()) {
            logger.info("Selected {} rows", logDataList.size());
        }

        return logDataList;
    }


    private Object[] getSqlBindings(LocalDateTime lastEventDatetime) {
        Timestamp startTimestamp = Timestamp.valueOf(lastEventDatetime);
        return new Object[]{startTimestamp};
    }

    /**
     * Если событий за одну секунду будет больше чем ROWS_FETCH_LIMIT, то джоба зациклится.
     * Определяем, что количество событий равно ROWS_FETCH_LIMIT и что последнее событие уже было обработано.
     * В этом случае нужно вручную сдвинуть LAST_EVENT_TIME_PPC_PROPERTY на 1 с.
     */
    private boolean isJobInLoop(List<ClientsFeaturesChangesLogData> logDataList, String lastEventUniqueId) {
        return logDataList.size() == ROWS_FETCH_LIMIT
                && lastEventUniqueId.equals(
                logDataList.get(ROWS_FETCH_LIMIT - 1).getUniqueId());
    }

    private void moveLastEventTimeByOneSecond(LocalDateTime lastEventDatetime) {
        ppcPropertiesSupport.get(LAST_EVENT_TIME_PPC_PROPERTY)
                .set(lastEventDatetime.plusSeconds(1));
    }

    /**
     * Запоминаем время и уникальный идентификатор последнего обработанного события.
     */
    private void updateLastEventDatetimeAndUniqueId(List<ClientsFeaturesChangesLogData> result) {
        if (result.isEmpty()) {
            return;
        }

        ClientsFeaturesChangesLogData lastLogData = result.get(result.size() - 1);
        logger.info("Last selected logData is: {}", lastLogData);
        ppcPropertiesSupport.get(LAST_EVENT_TIME_PPC_PROPERTY).set(lastLogData.getDateTime());
        ppcPropertiesSupport.get(LAST_EVENT_UNIQUE_ID_PPC_PROPERTY).set(lastLogData.getUniqueId());
    }

    private LocalDateTime getLastEventDatetime() {
        return ppcPropertiesSupport.get(LAST_EVENT_TIME_PPC_PROPERTY)
                .getOrDefault(now().minusDays(1));
    }

    private String getLastEventUniqueId() {
        return ppcPropertiesSupport.get(LAST_EVENT_UNIQUE_ID_PPC_PROPERTY)
                .getOrDefault("DEFAULT");
    }

    /**
     * Получить строку с идентификаторами фичей, для которых мы не хотим писать комментарии в тикеты.
     * Это фичи, которые слишком часто выдаются клиентам.
     * Результат: "'1','2','3'"
     */
    private String getIgnoredFeaturesIds() {
        List<Long> ignoredFeaturesIds = ppcPropertiesSupport.get(CLIENTS_FEATURES_CHANGES_LOG_JOB_IGNORED_FEATURES_IDS)
                .getOrDefault(List.of(5L, 77L, 427L, 453L, 663L, 679L));

        return ignoredFeaturesIds.stream()
                .map(featureId -> "'" + featureId + "'")
                .collect(joining(","));
    }

    /**
     * Проверяет, что джоба не отстает более чем на ALLOWED_HOURS_LAG часов.
     * Если джоба обрабатывает события, которые появились раньше чем ALLOWED_HOURS_LAG часов назад, то будет CRIT.
     */
    private void checkLag(List<ClientsFeaturesChangesLogData> fullLogDataList) {
        if (fullLogDataList.isEmpty()) {
            return;
        }

        LocalDateTime lastEventDateTime = Iterables.getLast(fullLogDataList).getDateTime();

        if (now().minusHours(ALLOWED_HOURS_LAG).isAfter(lastEventDateTime)) {
            logger.error("Job has lag of more than {} hours. Current processed date {}",
                    ALLOWED_HOURS_LAG, lastEventDateTime);
            setJugglerStatus(JugglerStatus.CRIT, "Job has lag. Current processed date: " + lastEventDateTime);
        }
    }

}
