package ru.yandex.direct.jobs.featureschanges;

import java.sql.Array;
import java.sql.SQLException;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jobs.featureschanges.model.ClientsFeaturesChangesLogData;
import ru.yandex.direct.jobs.featureschanges.model.FeaturesChangesSqlOperationType;
import ru.yandex.direct.jobs.featureschanges.model.FeaturesHistoryChangesLogData;

import static org.apache.commons.lang3.StringUtils.trimToNull;
import static ru.yandex.direct.dbschema.ppc.tables.ClientsFeatures.CLIENTS_FEATURES;
import static ru.yandex.direct.dbschema.ppcdict.tables.FeaturesHistory.FEATURES_HISTORY;
import static ru.yandex.direct.jobs.featureschanges.model.FeaturesChangesSqlOperationType.UNKNOWN;

public class FeaturesChangesLogMappers {
    private FeaturesChangesLogMappers() {
    }

    private static final Logger logger = LoggerFactory.getLogger(FeaturesChangesLogMappers.class);


    /**
     * Если в базе кривые данные, то не падаем, а игноируем такую запись.
     * Невозможно игнорировать невалидный dateTime, т.к. с помощью dateTime мы отмечаем обработанные записи
     * (с помощью lastEventDatetime в ppc_properties).
     */
    public static final RowMapper<ClientsFeaturesChangesLogData> CLIENTS_FEATURES_ROW_MAPPER =
            (rs, rowNum) -> {
                LocalDateTime dateTime = rs.getTimestamp(1).toLocalDateTime();
                String primaryKey = rs.getString(2);
                // На самом деле поле uniqueId не всегда будет уникально.
                // Но более подходящих полей, для того чтобы отличать события - нет.
                String uniqueId = dateTime + "-" + primaryKey;

                try {
                    String sqlOperationTypeRaw = rs.getString(3);
                    Array fieldNames = rs.getArray(4);
                    Array fieldValues = rs.getArray(5);
                    Long reqId = rs.getLong(6);
                    String method = trimToNull(rs.getString(7));

                    Map<String, String> nameToValue = getNameToValue(fieldNames, fieldValues);
                    String isEnabled = nameToValue.get(CLIENTS_FEATURES.IS_ENABLED.getName());

                    FeaturesChangesSqlOperationType sqlOperationType =
                            FeaturesChangesSqlOperationType.getEnum(sqlOperationTypeRaw);

                    if (sqlOperationType == UNKNOWN) {
                        logger.warn("Invalid row: {}", rs);
                        return ClientsFeaturesChangesLogData.createInvalidLogData(dateTime, uniqueId);
                    }

                    long[] compositeKey = splitCompositePrimaryKey(primaryKey);
                    ClientId clientId = ClientId.fromLong(compositeKey[0]);
                    long featureId = compositeKey[1];

                    return new ClientsFeaturesChangesLogData(
                            dateTime, sqlOperationType, isEnabled, clientId, featureId, reqId, uniqueId, method);
                } catch (RuntimeException e) {
                    logger.warn("Invalid row: {}", rs);
                    return ClientsFeaturesChangesLogData.createInvalidLogData(dateTime, uniqueId);
                }
            };


    /**
     * Если в базе кривые данные, то не падаем, а игноируем такую запись.
     * Невозможно игнорировать невалидный dateTime, т.к. с помощью dateTime мы отмечаем обработанные записи
     * (с помощью lastEventDatetime в ppc_properties).
     */
    public static final RowMapper<FeaturesHistoryChangesLogData> FEATURES_HISTORY_ROW_MAPPER =
            (rs, rowNum) -> {
                LocalDateTime dateTime = rs.getTimestamp(1).toLocalDateTime();
                String uniqueId = dateTime + "-" + rs.getString(4);

                try {
                    Array fieldNames = rs.getArray(2);
                    Array fieldValues = rs.getArray(3);
                    Map<String, String> nameToValue = getNameToValue(fieldNames, fieldValues);

                    Long featuresHistoryId = Long.valueOf(nameToValue.get(FEATURES_HISTORY.ID.getName()));
                    String featureTextId = nameToValue.get(FEATURES_HISTORY.FEATURE_TEXT_ID.getName());
                    Long operatorUid = Long.valueOf(nameToValue.get(FEATURES_HISTORY.OPERATOR_UID.getName()));

                    String sqlOperationTypeRaw = nameToValue.get(FEATURES_HISTORY.EVENT_TYPE.getName());
                    FeaturesChangesSqlOperationType sqlOperationType =
                            FeaturesChangesSqlOperationType.getEnum(sqlOperationTypeRaw);

                    return new FeaturesHistoryChangesLogData(
                            dateTime, sqlOperationType, nameToValue, featuresHistoryId, featureTextId, operatorUid,
                            uniqueId);
                } catch (RuntimeException e) {
                    logger.warn("Invalid row: {}", rs);
                    return FeaturesHistoryChangesLogData.createInvalidLogData(dateTime, uniqueId);
                }
            };

    private static Map<String, String> getNameToValue(Array fieldNames, Array fieldValues) throws SQLException {
        Map<String, String> nameToValue = new HashMap<>();
        String[] fieldNamesArray = (String[]) fieldNames.getArray();
        String[] fieldValuesArray = (String[]) fieldValues.getArray();
        for (int i = 0; i < fieldNamesArray.length; i++) {
            String fieldName = fieldNamesArray[i];
            String fieldValue = fieldValuesArray[i];
            nameToValue.put(fieldName, fieldValue);
        }
        return nameToValue;
    }

    /**
     * Вырезаем featureId из составного ключа таблицы clients_features
     */
    private static long[] splitCompositePrimaryKey(String compositeKey) {
        String[] split = compositeKey.split(",");
        if (split.length != 2) {
            return new long[]{0, 0};
        }

        long clientId = Long.parseLong(split[0].trim());
        long featureId = Long.parseLong(split[1].trim());

        return new long[]{clientId, featureId};
    }

}
