package ru.yandex.direct.jobs.featureschanges;

import java.sql.Timestamp;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.featureschanges.handlers.FeaturesChangesLogHandler;
import ru.yandex.direct.jobs.featureschanges.handlers.FeaturesChangesLogHandlerFactory;
import ru.yandex.direct.jobs.featureschanges.model.FeaturesHistoryChangesLogData;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.startrek.client.error.StartrekInternalClientError;
import ru.yandex.startrek.client.error.StartrekInternalServerError;
import ru.yandex.startrek.client.error.UnauthorizedException;

import static java.time.LocalDateTime.now;
import static ru.yandex.direct.common.db.PpcPropertyType.LOCAL_DATE_TIME;
import static ru.yandex.direct.common.db.PpcPropertyType.STRING;
import static ru.yandex.direct.dbutil.wrapper.SimpleDb.CLICKHOUSE_CLOUD;
import static ru.yandex.direct.jobs.featureschanges.FeaturesChangesLogFilterHelper.getFirstUnprocessedLogData;
import static ru.yandex.direct.jobs.featureschanges.FeaturesChangesLogMappers.FEATURES_HISTORY_ROW_MAPPER;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Джоба следит за обновлениями в таблицах features_history и записывает изменения в стартрек.
 * Один запуск джобы может работать только с одним тикетом.
 */

@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10),
        tags = {DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM},
        needCheck = ProductionOnly.class
)
@Hourglass(periodInSeconds = 30, needSchedule = ProductionOnly.class)
public class FeaturesHistoryChangesLogJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(FeaturesHistoryChangesLogJob.class);
    private static final int ROWS_FETCH_LIMIT = 100;

    private static final PpcPropertyName<LocalDateTime> LAST_EVENT_TIME_PPC_PROPERTY =
            new PpcPropertyName<>("features_bin_logs_reader.features_history_last_event_datetime", LOCAL_DATE_TIME);

    private static final PpcPropertyName<String> LAST_EVENT_UNIQUE_ID_PPC_PROPERTY =
            new PpcPropertyName<>("features_bin_logs_reader.features_history_last_event_unique_id", STRING);

    private static final String SQL_TEMPLATE =
            "SELECT datetime, row.name, row.value, primary_key" +
                    "        FROM binlog_rows_v2 as bnlgs " +
                    "       WHERE datetime >= ? " +
                    // события с разных шардов могут приходить не в порядке datetime (приходить с отставанием)
                    "         AND datetime < minus(now(), toIntervalMinute(5)) " +
                    "         AND table='features_history' " +
                    "    ORDER BY datetime, primary_key" +
                    "       LIMIT " + ROWS_FETCH_LIMIT;


    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final DatabaseWrapperProvider dbProvider;
    private final FeaturesChangesLogHandlerFactory handlerFactory;
    private final ShardHelper shardHelper;

    @Autowired
    public FeaturesHistoryChangesLogJob(PpcPropertiesSupport ppcPropertiesSupport, DatabaseWrapperProvider dbProvider,
                                        FeaturesChangesLogHandlerFactory handlerFactory, ShardHelper shardHelper) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.dbProvider = dbProvider;
        this.handlerFactory = handlerFactory;
        this.shardHelper = shardHelper;
    }

    @Override
    public void execute() {
        String lastEventUniqueId = getLastEventUniqueId();
        LocalDateTime lastEventDatetime = getLastEventDatetime();
        List<FeaturesHistoryChangesLogData> logDataList = selectRows(lastEventDatetime);

        if (isJobInLoop(logDataList, lastEventUniqueId)) {
            logger.warn("Job in loop. Move last_event_datetime by one second");
            moveLastEventTimeByOneSecond(lastEventDatetime);
            return;
        }

        Optional<FeaturesHistoryChangesLogData> logDataOptional =
                getFirstUnprocessedLogData(logDataList, lastEventUniqueId);

        if (logDataOptional.isEmpty()) {
            return;
        }

        FeaturesHistoryChangesLogData logData = logDataOptional.get();
        if (logData.isBrokenData()) {
            // сразу сдвигаем lastEventDatetime
            updateLastEventDatetimeAndUniqueId(logData);
            return;
        }

        handleLogData(logData);
    }

    private void handleLogData(FeaturesHistoryChangesLogData logData) {
        try {
            enrichWithOperatorLogin(logData);
            FeaturesChangesLogHandler handler = handlerFactory.createFeaturesHistoryHandler(logData);
            handler.handle();
        } catch (StartrekInternalClientError | StartrekInternalServerError | UnauthorizedException e) {
            //если стартрек недоступен, то джоба должна обработать эти события еще раз, при следующем запуске.
            logger.info("An error occurred while accessing to the Startrek");
            throw e;
        } catch (RuntimeException suppressed) {
            // любые другие ошибки подавляем и обновляем LastEventDatetime
            logger.warn("Unable to handle events", suppressed);
        }

        updateLastEventDatetimeAndUniqueId(logData);
    }

    private void enrichWithOperatorLogin(FeaturesHistoryChangesLogData logData) {
        Long operatorUid = logData.getOperatorUid();
        if (operatorUid == null || operatorUid == 0) {
            logger.warn("Operator uid is not set for {}", logData);
            return;
        }

        try {
            String operatorLogin = shardHelper.getLoginByUid(operatorUid);
            logData.setLogin(operatorLogin);
        } catch (RuntimeException e) {
            logger.warn("Unable to get operator login for {}", logData);
        }
    }

    private List<FeaturesHistoryChangesLogData> selectRows(LocalDateTime lastEventDatetime) {
        Object[] bindings = getSqlBindings(lastEventDatetime);
        List<FeaturesHistoryChangesLogData> logDataList = dbProvider.get(CLICKHOUSE_CLOUD)
                .query(SQL_TEMPLATE, bindings, FEATURES_HISTORY_ROW_MAPPER);

        if (!logDataList.isEmpty()) {
            logger.info("Selected {} rows", logDataList.size());
        }

        return logDataList;
    }


    private Object[] getSqlBindings(LocalDateTime lastEventDatetime) {
        Timestamp startTimestamp = Timestamp.valueOf(lastEventDatetime);
        return new Object[]{startTimestamp};
    }

    /**
     * Если событий за одну секунду будет больше чем ROWS_FETCH_LIMIT, то джоба зациклится.
     * Определяем, что количество событий равно ROWS_FETCH_LIMIT и что последнее событие уже было обработано.
     * В этом случае нужно вручную сдвинуть LAST_EVENT_TIME_PPC_PROPERTY на 1 с.
     */
    private boolean isJobInLoop(List<FeaturesHistoryChangesLogData> logDataList, String lastEventUniqueId) {
        return logDataList.size() == ROWS_FETCH_LIMIT
                && lastEventUniqueId.equals(
                logDataList.get(ROWS_FETCH_LIMIT - 1).getUniqueId());
    }

    private void moveLastEventTimeByOneSecond(LocalDateTime lastEventDatetime) {
        ppcPropertiesSupport.get(LAST_EVENT_TIME_PPC_PROPERTY)
                .set(lastEventDatetime.plusSeconds(1));
    }

    /**
     * Запоминаем время и уникальный идентификатор последнего обработанного события.
     */
    private void updateLastEventDatetimeAndUniqueId(FeaturesHistoryChangesLogData lastLogData) {

        logger.info("Last selected logData is: {}", lastLogData);
        ppcPropertiesSupport.get(LAST_EVENT_TIME_PPC_PROPERTY).set(lastLogData.getDateTime());
        ppcPropertiesSupport.get(LAST_EVENT_UNIQUE_ID_PPC_PROPERTY).set(lastLogData.getUniqueId());
    }

    private LocalDateTime getLastEventDatetime() {
        return ppcPropertiesSupport.get(LAST_EVENT_TIME_PPC_PROPERTY)
                .getOrDefault(now().minusDays(1));
    }

    private String getLastEventUniqueId() {
        return ppcPropertiesSupport.get(LAST_EVENT_UNIQUE_ID_PPC_PROPERTY)
                .getOrDefault("DEFAULT");
    }

}
