package ru.yandex.direct.jobs.featureschanges;


import java.util.Collection;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.jobs.export.feature.diff.FeatureRow;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.ComponentRef;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueCreate;
import ru.yandex.startrek.client.model.IssueRef;
import ru.yandex.startrek.client.model.IssueUpdate;
import ru.yandex.startrek.client.model.UserRef;

import static java.util.Collections.singleton;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nullableNvl;


@Service
public class FeaturesLogStartrekService {

    private static final Logger logger = LoggerFactory.getLogger(FeaturesLogStartrekService.class);

    public static final String DEAD_FEATURE_TAG = "dead_feature";
    private static final String QUEUE = "DIRECT";
    private static final String TICKET_TYPE = "task";
    private static final String MAIN_COMPONENT_NAME = "FeatureFlag";
    private static final String FIXED_STATUS = "fixed";
    private static final String CLOSE_TRANSITION = "close";
    private static final String SUMMARY_TEMPLATE = "Feature: %s";
    private static final Integer TS_FEATURE_BOARD = 13475;
    private static final String TS_FEATURE_TAG = "ts-feature";

    private final String startrekUrl;
    private final Session session;

    @Autowired
    public FeaturesLogStartrekService(EnvironmentType environmentType,
                                      @Value("startrek.startrek_url") String startrekUrl,
                                      Session session) {
        if (environmentType.equals(EnvironmentType.PRODUCTION)) {
            this.startrekUrl = startrekUrl;
        } else {
            this.startrekUrl = "https://st.test.yandex-team.ru";
        }
        this.session = session;
    }

    /**
     * Создать основной сторожевой тикет
     */
    public String createMainTicket(String featureName, String owner, Collection<String> watchers) {

        String ticketKey = getTicketKey(featureName);
        if (ticketKey != null) {
            logger.warn("Issue already exists: {}", getTicketUrl(ticketKey));
            return ticketKey;
        }

        IssueCreate issueCreate = IssueCreate.builder()
                .queue(QUEUE)
                .type(TICKET_TYPE)
                .summary(String.format(SUMMARY_TEMPLATE, featureName))
                .description("//Тикет создан с помощью " + FeaturesHistoryChangesLogJob.class.getSimpleName() + "//")
                .components(getMainComponentRef())
                .assignee(owner)
                .followers(watchers == null ? null : watchers.toArray(new String[0]))
                .build();

        Issue issue = session.issues().create(issueCreate);
        logger.info("Main issue created: {} for feature {}", getTicketUrl(issue.getKey()), featureName);
        return issue.getKey();
    }

    /**
     * Создать тикет на включение фичи на ТС
     */
    public Issue createTsTicket(String summary, String description, FeatureRow prodFeature) {
        String featureChanger = prodFeature.getLogin();
        String assignee = nullableNvl(prodFeature.getOwner(), featureChanger);
        String featureTextId = prodFeature.getTextId();

        IssueCreate.Builder issueBuilder = IssueCreate.builder()
                .queue(QUEUE)
                .type(TICKET_TYPE)
                .author("ppc")
                .summary(summary)
                .description(description)
                .assignee(assignee)
                .tags(Cf.toList(List.of(TS_FEATURE_TAG, featureTextId)))
                .set("boards", Cf.toList(singleton(TS_FEATURE_BOARD)));

        if (featureChanger != null) {
            issueBuilder.followers(featureChanger);
        }

        String parentTicketId = prodFeature.getTicketId();
        if (parentTicketId != null) {
            issueBuilder.parent(parentTicketId);
        }

        Issue issue = session.issues().create(issueBuilder.build());
        logger.info("created ts issue {} for feature {}", getTicketUrl(issue.getKey()), featureTextId);
        return issue;
    }

    public String getTicketKey(String featureName) {
        Issue ticket = getTicket(featureName);
        return ifNotNull(ticket, IssueRef::getKey);
    }

    public Issue getTicket(String featureName) {
        String query =
                String.format("\"Queue\": \"%s\" " +
                                "AND \"Summary\": \"Feature: %s\" " +
                                "AND \"Components\": \"%s\"",
                        QUEUE, featureName, MAIN_COMPONENT_NAME);

        String expectedSummary = String.format(SUMMARY_TEMPLATE, featureName);

        List<Issue> issueList = session.issues().find(query).stream()
                .filter(issue -> expectedSummary.equals(issue.getSummary()))
                .collect(toList());

        if (issueList.size() > 1) {
            logger.warn("Multiple tickets with same key {}, search result: {}", featureName, issueList);
        }

        return issueList.isEmpty() ? null : issueList.get(0);
    }

    public void addTicketComment(String ticketKey, String text) {
        Issue issue = session.issues().get(ticketKey);
        issue.comment(text);
        logger.info("Added comment to issue {}", getTicketUrl(ticketKey));
    }

    /**
     * Добавить комментарий с призывом исполнителя
     */
    public void addTicketCommentWithAssigneeNotification(String ticketKey, String text) {
        Issue issue = session.issues().get(ticketKey);
        UserRef[] assignee = issue.getAssignee().toArray(UserRef.class);
        issue.comment(text, assignee);
        logger.info("Added comment to issue {}", getTicketUrl(ticketKey));
    }

    public void addTicketTag(String ticketKey, String tag) {
        IssueUpdate issueUpdate = IssueUpdate.builder()
                .tags(Cf.arrayList(tag), Cf.arrayList())
                .build();

        session.issues().update(ticketKey, issueUpdate);
        logger.info("Added tag to issue {}", getTicketUrl(ticketKey));
    }

    public void closeTicket(String ticketKey, String text) {
        IssueUpdate issueUpdate = IssueUpdate.builder()
                .resolution(FIXED_STATUS)
                .removeTags(DEAD_FEATURE_TAG)
                .comment(CommentCreate
                        .comment(text)
                        .build())
                .build();

        session.issues().update(ticketKey, issueUpdate);
        session.transitions().execute(ticketKey, CLOSE_TRANSITION);
        logger.info("Ticket closed {}", getTicketUrl(ticketKey));
    }

    /**
     * Достать тикеты на включение фичи на ТС
     */
    public List<Issue> findTsFeatureTickets(String featureTextId) {
        String query = String.format("Queue: \"%s\" " +
                        "AND \"Tags\": \"%s\" " +
                        "AND \"Tags\": \"%s\" ",
                QUEUE, TS_FEATURE_TAG, featureTextId);

        return session.issues().find(query).collect(toList());
    }

    public String getTicketUrl(String ticketKey) {
        return this.startrekUrl + "/" + ticketKey;
    }

    private ComponentRef getMainComponentRef() {
        return session.queues().get(QUEUE).getComponents().stream()
                .filter(c -> MAIN_COMPONENT_NAME.equalsIgnoreCase(c.getDisplay()))
                .findAny()
                .orElseThrow(() -> new RuntimeException("Can't get ComponentRef for " + MAIN_COMPONENT_NAME));
    }

}
