package ru.yandex.direct.jobs.featureschanges;

import java.sql.Date;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.jobs.featureschanges.model.ClientsFeaturesChangesLogData;
import ru.yandex.direct.jobs.featureschanges.model.PpcLogCmdUserData;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.dbutil.wrapper.SimpleDb.CLICKHOUSE_CLOUD;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;

@Service
public class PpcLogCmdUserDataService {

    private static final int ROWS_FETCH_LIMIT = 10000;
    private static final Logger logger = LoggerFactory.getLogger(PpcLogCmdUserDataService.class);
    private static final RowMapper<PpcLogCmdUserData> ROW_MAPPER =
            (rs, rowNum) -> new PpcLogCmdUserData(rs.getLong(1), rs.getString(2), rs.getLong(3));
    private static final String REQ_IDS_PLACEHOLDER = "{REQ_IDS}";

    private static final String SQL_TEMPLATE =
            "SELECT  cmdlg.uid, cmdlg.service, cmdlg.reqid" +
                    "        FROM ppclog_cmd as cmdlg" +
                    "       WHERE cmdlg.log_date > ? " +
                    "         AND cmdlg.reqid in (" + REQ_IDS_PLACEHOLDER + ") " +
                    "       LIMIT " + ROWS_FETCH_LIMIT;

    private final DatabaseWrapperProvider dbProvider;
    private final UserRepository userRepository;

    @Autowired
    public PpcLogCmdUserDataService(DatabaseWrapperProvider dbProvider, UserRepository userRepository) {
        this.dbProvider = dbProvider;
        this.userRepository = userRepository;
    }

    public void enrichWithUserData(List<ClientsFeaturesChangesLogData> logDataList, LocalDateTime lastEventDatetime) {

        Set<Long> reqIds = mapAndFilterToSet(logDataList, ClientsFeaturesChangesLogData::getReqId, Objects::nonNull);
        List<PpcLogCmdUserData> userDataList = selectRows(lastEventDatetime, reqIds);

        Map<Long, String> loginsByUids = getLoginsByUids(userDataList);

        enrich(logDataList, loginsByUids, userDataList);
    }

    private Map<Long, String> getLoginsByUids(List<PpcLogCmdUserData> userDataList) {
        Set<Long> uids = mapAndFilterToSet(userDataList, PpcLogCmdUserData::getUid, uid -> uid != 0);
        return userRepository.getLoginsByUids(uids);
    }

    private List<PpcLogCmdUserData> selectRows(LocalDateTime lastEventDatetime, Set<Long> reqIds) {
        if (reqIds.isEmpty()) {
            return emptyList();
        }

        Object[] bindings = getSqlBindings(lastEventDatetime);
        // подставляем reqIds в запрос
        String reqIdsString = reqIds.stream().map(Object::toString).collect(joining(","));
        String sql = SQL_TEMPLATE.replace(REQ_IDS_PLACEHOLDER, reqIdsString);

        List<PpcLogCmdUserData> userDataList = dbProvider.get(CLICKHOUSE_CLOUD)
                .query(sql, bindings, ROW_MAPPER);
        logger.info("Selected {} rows", userDataList.size());
        return userDataList;
    }

    private void enrich(List<ClientsFeaturesChangesLogData> logDataList,
                        Map<Long, String> loginsByUids,
                        List<PpcLogCmdUserData> userDataList) {

        Map<Long, PpcLogCmdUserData> reqIdToUserData = userDataList.stream()
                .collect(toMap(PpcLogCmdUserData::getReqId, identity(), (a, b) -> a));

        logDataList.forEach(logData -> {

            Long reqId = logData.getReqId();
            PpcLogCmdUserData ppcLogCmdUserData = reqIdToUserData.get(reqId);
            if (ppcLogCmdUserData == null) {
                return;
            }

            Long uid = ppcLogCmdUserData.getUid();
            String login = loginsByUids.getOrDefault(uid, ppcLogCmdUserData.getServiceName());
            logData.setLogin(login);
        });
    }


    private Object[] getSqlBindings(LocalDateTime lastEventDatetime) {
        //Берем с запасом, т.к. время в binlog_rows_v2 и ppclog_cmd может не совпадать
        Date ppcLogCmdStartData = Date.valueOf(lastEventDatetime.toLocalDate().minusDays(1));
        return new Object[]{ppcLogCmdStartData};
    }
}
