package ru.yandex.direct.jobs.featureschanges

import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.feature.model.EventType
import ru.yandex.direct.core.entity.feature.model.Feature
import ru.yandex.direct.core.entity.feature.model.FeatureHistory
import ru.yandex.direct.core.entity.feature.repository.FeatureHistoryRepository
import ru.yandex.direct.core.entity.feature.repository.FeatureRepository
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.jobs.featureschanges.FeaturesCommentUtils.getFeaturesInCodeText
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import java.time.LocalDateTime
import java.time.temporal.ChronoUnit

private const val ONE_HUNDRED_PERCENT = 100

private const val FIRST_REMIND_IN_DAYS = 30L
private const val SECOND_REMIND_IN_DAYS = 45L;
private const val REMIND_EVERY_DAYS = 7L

private const val DEAD_FEATURE_EXPIRED_TAG = "dead_feature_expired";

@JugglerCheck(ttl = JugglerCheck.Duration(days = 2),
    tags = [CheckTag.DIRECT_PRIORITY_2, CheckTag.DIRECT_PRODUCT_TEAM], needCheck = ProductionOnly::class)
@Hourglass(cronExpression = "0 0 1 * * ?", needSchedule = ProductionOnly::class)
class RemindToDeleteFeatureJob(
    private val featureHistoryRepository: FeatureHistoryRepository,
    private val featureRepository: FeatureRepository,
    private val startrekService: FeaturesLogStartrekService
) : DirectJob() {

    private val logger = LoggerFactory.getLogger(RemindToDeleteFeatureJob::class.java)


    override fun execute() {
        val prodFeaturesMap = readFeatureTable()
        val launchedFeaturesIds = prodFeaturesMap.values
            .asSequence()
            .filter { feature -> feature.settings.percent == ONE_HUNDRED_PERCENT }
            .map(Feature::getId)
            .toList()

        val featureHistoryMap = featureHistoryRepository.getFeatureIdToHistory(launchedFeaturesIds)

        featureHistoryMap.forEach { (_, featureUpdates) ->
            val featureHistory = featureUpdates
                .asSequence()
                .filter { featureHistory -> featureHistory.eventType == EventType.UPDATE }
                .filter { featureHistory -> featureHistory.newSettings.percent == ONE_HUNDRED_PERCENT }
                .sortedWith(Comparator.comparing(FeatureHistory::getEventTime))
                .firstOrNull()

            if (featureHistory != null &&
                prodFeaturesMap.containsKey(featureHistory.featureTextId)) {
                val daysSinceFeatureLaunch = daysSince(featureHistory.eventTime)
                if (shouldRemind(daysSinceFeatureLaunch)) {
                    val feature = prodFeaturesMap[featureHistory.featureTextId]
                    remindToDeleteFeature(feature!!, daysSinceFeatureLaunch)
                }
            }
        }
    }

    private fun readFeatureTable(): Map<String, Feature> {
        return featureRepository.get()
            .associateBy(Feature::getFeatureTextId)
    }

    private fun remindToDeleteFeature(feature: Feature, daysSinceFeatureLaunch: Long) {
        val featureTicketId = feature.featureTicket
        val featureTextId = feature.featureTextId;
        if (featureTicketId == null) {
            logger.error("ticket for feature {} does not exist", featureTextId)
            return
        }

        val comment = "Пожалуйста, удалите фичу \"$featureTextId\" и закройте этот тикет.\n" +
            getFeaturesInCodeText(featureTextId) + "\n" +
            "Фича была запущена $daysSinceFeatureLaunch дней назад." +
            "----\n" +
            "Комментарий оставлен джобой RemindToDeleteFeatureJob из приложения jobs"

        startrekService.addTicketCommentWithAssigneeNotification(featureTicketId, comment)
        startrekService.addTicketTag(featureTicketId, DEAD_FEATURE_EXPIRED_TAG)
        logger.info("updated issue {} for feature {}", featureTicketId, featureTextId)

    }

    private fun daysSince(localDateTime: LocalDateTime): Long {
        return ChronoUnit.DAYS.between(localDateTime, LocalDateTime.now())
    }

    private fun shouldRemind(daysSinceFeatureLaunch: Long): Boolean {
        return daysSinceFeatureLaunch == FIRST_REMIND_IN_DAYS ||
            daysSinceFeatureLaunch == SECOND_REMIND_IN_DAYS ||
            (daysSinceFeatureLaunch > SECOND_REMIND_IN_DAYS &&
                0L == (daysSinceFeatureLaunch - SECOND_REMIND_IN_DAYS) % REMIND_EVERY_DAYS)
    }
}
