package ru.yandex.direct.jobs.featureschanges.handlers;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.core.entity.feature.repository.FeatureRepository;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.jobs.featureschanges.FeaturesLogStartrekService;
import ru.yandex.direct.jobs.featureschanges.exception.TicketNotFound;
import ru.yandex.direct.jobs.featureschanges.model.ClientsFeaturesChangesLogData;

import static java.util.Collections.singleton;
import static java.util.stream.Collectors.groupingBy;
import static ru.yandex.direct.jobs.featureschanges.model.FeaturesChangesSqlOperationType.DELETE;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Все обрабатываемые события относятся к одной фиче.
 */
public class ClientsFeaturesAllLogHandler implements FeaturesChangesLogHandler {

    private static final Logger logger = LoggerFactory.getLogger(ClientsFeaturesAllLogHandler.class);
    private static final String FEATURE_ON = "включил";
    private static final String FEATURE_OFF = "выключил";
    private static final String FEATURE_ON_NUM = "1";
    private static final int MAX_EVENTS_BY_ONE_LOGIN = 30;

    private final List<ClientsFeaturesChangesLogData> logDataList;
    private final FeaturesLogStartrekService startrekService;
    private final FeatureRepository featureRepository;
    private final UserService userService;

    public ClientsFeaturesAllLogHandler(List<ClientsFeaturesChangesLogData> logDataList,
                                        FeaturesLogStartrekService startrekService,
                                        FeatureRepository featureRepository, UserService userService) {
        this.logDataList = logDataList;
        this.startrekService = startrekService;
        this.featureRepository = featureRepository;
        this.userService = userService;
    }

    @Override
    public void handle() {
        if (logDataList.isEmpty()) {
            return;
        }

        Feature feature = getFeature();
        String featureTextId = feature.getFeatureTextId();

        String ticketKey = feature.getFeatureTicket();

        // События о создании фичи обрабатываются с задержкой. Поэтому падаем, пока тикет не будет создан.
        if (ticketKey == null) {
            throw new TicketNotFound(featureTextId);
        }

        String comment = getComment();
        startrekService.addTicketComment(ticketKey, comment);
    }

    private String getComment() {

        Set<ClientId> clientIds = listToSet(logDataList, ClientsFeaturesChangesLogData::getClientId);
        Map<ClientId, String> chiefsLoginsByClientIds = userService.getChiefsLoginsByClientIds(clientIds);

        Map<String, List<ClientsFeaturesChangesLogData>> operatorToChanges =
                logDataList.stream().collect(
                        groupingBy(logData -> logData.getLogin() + " " + getFeatureState(logData)));

        StringBuilder commentBuilder = new StringBuilder();

        operatorToChanges.forEach((operatorWithFeatureState, changesForOneOperator) -> {
            int maxSize = Math.min(changesForOneOperator.size(), MAX_EVENTS_BY_ONE_LOGIN);
            commentBuilder
                    .append(operatorWithFeatureState)
                    .append(String.format(" фичу %s клиентам:", changesForOneOperator.size()))
                    .append("\n");

            changesForOneOperator.subList(0, maxSize).forEach(logData ->
                    commentBuilder.append("- ClientID = ")
                            .append(logData.getClientId())
                            .append(" (").append(chiefsLoginsByClientIds.get(logData.getClientId())).append(")")
                            .append("\n")
            );

            if (maxSize < changesForOneOperator.size()) {
                commentBuilder.append("<...>\n");
            }
        });

        return commentBuilder.toString();
    }

    private String getFeatureState(ClientsFeaturesChangesLogData logData) {
        if (logData.getSqlOperationType() == DELETE) {
            return FEATURE_OFF;
        } else {
            String isEnabled = logData.getIsEnabled();
            if (isEnabled == null) {
                logger.warn("Unexpected empty is_enabled field {}", logData);
                return null;
            }

            if (FEATURE_ON_NUM.equalsIgnoreCase(isEnabled)) {
                return FEATURE_ON;
            } else {
                return FEATURE_OFF;
            }
        }
    }

    private Feature getFeature() {
        Long featureId = logDataList.get(0).getFeatureId();
        return featureRepository.get(singleton(featureId)).get(0);
    }
}
