package ru.yandex.direct.jobs.featureschanges.handlers;

import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.core.entity.feature.model.FeatureSettings;
import ru.yandex.direct.core.entity.feature.repository.FeatureHistoryRepository;
import ru.yandex.direct.core.entity.feature.repository.FeatureRepository;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.feature.FeatureType;
import ru.yandex.direct.jobs.featureschanges.FeaturesLogStartrekService;
import ru.yandex.direct.jobs.featureschanges.exception.TicketNotFound;
import ru.yandex.direct.jobs.featureschanges.model.FeaturesHistoryChangesLogData;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.core.entity.feature.repository.FeatureHistoryRepository.EMPTY_FEATURE_SETTINGS;
import static ru.yandex.direct.jobs.featureschanges.FeaturesCommentUtils.getFeaturesInCodeText;
import static ru.yandex.direct.jobs.featureschanges.FeaturesLogStartrekService.DEAD_FEATURE_TAG;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

public class FeaturesHistoryUpdateLogHandler implements FeaturesChangesLogHandler {

    private static final Logger logger = LoggerFactory.getLogger(FeaturesHistoryUpdateLogHandler.class);
    private static final String HUNDRED_PERCENT_MESSAGE =
            "Процент изменился на 100%. Пожалуйста, не забудьте удалить фичу из кода.\n";

    private final FeaturesHistoryChangesLogData logData;
    private final FeaturesLogStartrekService startrekService;
    private final FeatureHistoryRepository repository;
    private final FeatureRepository featureRepository;

    public FeaturesHistoryUpdateLogHandler(FeaturesHistoryChangesLogData logData,
                                           FeaturesLogStartrekService startrekService,
                                           FeatureHistoryRepository featureHistoryRepository,
                                           FeatureRepository featureRepository) {
        this.startrekService = startrekService;
        this.logData = logData;
        this.repository = featureHistoryRepository;
        this.featureRepository = featureRepository;
    }

    @Override
    public void handle() {
        String featureTextId = logData.getFeatureTextId();
        Optional<Feature> featureOptional = featureRepository.getByFeatureTextId(featureTextId);
        String ticketKey = featureOptional
                .map(Feature::getFeatureTicket)
                .orElseThrow(() -> new TicketNotFound(featureTextId));

        FeatureSettings prevSettings = getPreviousFeatureSettings();
        FeatureSettings currentSettings = getCurrentFeatureSettings();
        String diff = diffBetweenSettings(currentSettings, prevSettings);
        if (diff == null) {
            logger.warn("Изменения не найдены: {}", logData);
            return;
        }

        startrekService.addTicketComment(ticketKey, diff);
        if (becameHundredPercent(prevSettings, currentSettings) && isNotPermanentFeature(featureTextId)) {
            startrekService.addTicketTag(ticketKey, DEAD_FEATURE_TAG);
            startrekService.addTicketComment(ticketKey, HUNDRED_PERCENT_MESSAGE + getFeaturesInCodeText(featureTextId));
        }
    }

    private boolean isNotPermanentFeature(String featureTextId) {
        FeatureName featureName = FeatureName.fromString(featureTextId.toUpperCase());
        FeatureType featureType = ifNotNull(featureName, FeatureName::getFeatureType);
        return featureType != FeatureType.PERMANENT;
    }

    private boolean becameHundredPercent(FeatureSettings prevSettings, FeatureSettings currentSettings) {
        return currentSettings.getPercent() == 100 && prevSettings.getPercent() != 100;
    }

    private FeatureSettings getPreviousFeatureSettings() {
        Long featuresHistoryId = logData.getFeaturesHistoryId();
        String featureTextId = logData.getFeatureTextId();
        String plainFeatureSettingsJson =
                repository.getPrevPlainFeatureSettings(featureTextId, featuresHistoryId);
        return getFeatureSettingsFromJson(plainFeatureSettingsJson);
    }

    private FeatureSettings getCurrentFeatureSettings() {
        Long featuresHistoryId = logData.getFeaturesHistoryId();
        String plainFeatureSettingsJson = repository.getPlainFeatureSettings(featuresHistoryId);
        return getFeatureSettingsFromJson(plainFeatureSettingsJson);
    }

    /**
     * FeatureMappings.fromDb не используется, т.к. в нем невозможно отличить состояние, когда настройки не были заданы
     * <p>
     * При изменении полей в таблице features не относящихся к полю settings, поле features_history.new_settings
     * заполняется пустым значением - "{}"
     */
    private FeatureSettings getFeatureSettingsFromJson(String plainFeatureSettingsJson) {
        if (plainFeatureSettingsJson == null || plainFeatureSettingsJson.equals(EMPTY_FEATURE_SETTINGS)) {
            return null;
        }
        return JsonUtils.fromJson(plainFeatureSettingsJson, FeatureSettings.class);
    }

    /**
     * Сравнивем все ModelProperty между currentSettings и prevSettings.
     * Если изменений нет, то возвращаем null
     */
    private String diffBetweenSettings(FeatureSettings currentSettings, FeatureSettings prevSettings) {
        // В ситуациях, когда настройки в features_history не заполняются, не пытаемся сравнивать.
        if (currentSettings == null || prevSettings == null) {
            return null;
        }

        StringBuilder diffBuilder = new StringBuilder("**" + logData.getLogin() + "** обновил состояние фичи:\n");
        AtomicBoolean nothingChanged = new AtomicBoolean(true);

        FeatureSettings.allModelProperties().forEach(modelProperty -> {
            ModelProperty<FeatureSettings, Object> typedModelProperty =
                    (ModelProperty<FeatureSettings, Object>) modelProperty;

            Object curValue = typedModelProperty.get(currentSettings);
            Object prevValue = typedModelProperty.get(prevSettings);
            if (!Objects.equals(curValue, prevValue)) {
                nothingChanged.set(false);
                diffBuilder.append("- **").append(typedModelProperty.name())
                        .append("** : !!").append(prevValue)
                        .append("!! => !!(green)").append(curValue).append("!!\n");
            }
        });

        if (nothingChanged.get()) {
            return null;
        }

        return diffBuilder.toString();
    }
}
