package ru.yandex.direct.jobs.featuresync;

import java.sql.SQLException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToClientIdState;
import ru.yandex.direct.core.entity.feature.container.LoginClientIdChiefLoginWithState;
import ru.yandex.direct.core.entity.feature.model.FeatureState;
import ru.yandex.direct.core.entity.feature.service.FeatureCache;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.yql.ResultSetFuture;
import ru.yandex.yql.YqlConnection;
import ru.yandex.yql.YqlPreparedStatement;

import static java.util.concurrent.TimeUnit.MINUTES;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING;
import static ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion.SQLv1;

/**
 * Синхронизирует список клиентов, которым НЕ нужно открывать чат поддержки, с выгрузкой в YT
 */
@Hourglass(periodInSeconds = 60 * 60, needSchedule = ProductionOnly.class)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3, minutes = 5),
        tags = {DIRECT_PRIORITY_2, CheckTag.GROUP_INTERNAL_SYSTEMS},
        notifications = {@OnChangeNotification(recipient = CHAT_INTERNAL_SYSTEMS_MONITORING,
                status = {JugglerStatus.OK, JugglerStatus.CRIT},
                method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class CheckedSupportChatSyncJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(CheckedSupportChatSyncJob.class);
    private static final YtCluster YT_CLUSTER = YtCluster.HAHN;
    private static final YtTable TABLE =
            new YtTable("//home/oobp/public/clients_chat_exclusion");
    private static final PpcPropertyName<LocalDateTime> SYNCED_MOD_TIME_PROPERTY =
            PpcPropertyNames.CHECKED_SUPPORT_CHAT_FEATURE_LAST_SYNCED_MOD_TIME;
    private static final String YQL_QUERY = String.join("\n",
            new ClassPathResourceInputStreamSource("feature-sync/support_chat.yql").readLines());
    private static final long YQL_QUERY_TIMEOUT_MINUTES = 15L;
    private static final long MIN_ROW_COUNT = 500L;
    private static final int YT_TABLE_ACCEPTABLE_AGE_IN_DAYS = 3;
    static final String FEATURE_NAME = FeatureName.CHECKED_SUPPORT_CHAT.getName();

    private final YtProvider ytProvider;
    private final PpcProperty<LocalDateTime> property;
    private final FeatureManagingService service;
    private final ShardHelper helper;
    private final FeatureCache featureCache;

    private Long featureId;
    private LocalDateTime modTime;

    @Autowired
    public CheckedSupportChatSyncJob(YtProvider ytProvider,
                                     PpcPropertiesSupport propertiesSupport,
                                     ShardHelper helper,
                                     FeatureCache featureCache,
                                     FeatureManagingService service) {
        this(ytProvider, propertiesSupport.get(SYNCED_MOD_TIME_PROPERTY), helper, featureCache, service);
    }

    CheckedSupportChatSyncJob(YtProvider ytProvider,
                              PpcProperty<LocalDateTime> property,
                              ShardHelper helper,
                              FeatureCache featureCache,
                              FeatureManagingService service) {
        this.ytProvider = ytProvider;
        this.property = property;
        this.featureCache = featureCache;
        this.service = service;
        this.helper = helper;
    }

    @Override
    public void execute() {
        featureId = featureCache.getIdsByTextId(List.of(FEATURE_NAME)).get(FEATURE_NAME);
        if (featureId == null) {
            logger.error("Feature id for {} was not found", FEATURE_NAME);
            return;
        }

        YtOperator ytOperator = ytProvider.getOperator(YtCluster.HAHN);
        if (!shouldSync(property.getOrDefault(LocalDateTime.MIN), ytOperator)) {
            return;
        }

        YPath tableDir;
        try (TraceProfile ignore = Trace.current().profile("support_chat:yql")) {
            tableDir = createDiffTables();
        } catch (SQLException | ExecutionException | TimeoutException e) {
            logger.error("Got an YT query exception", e);
            return;
        }

        ClientIdTableRow tablerow = new ClientIdTableRow();
        YtTable turnOnTable = new YtTable(tableDir + "/turn_on");
        long countTurnOn = ytOperator.readTableSnapshot(turnOnTable,
                tablerow,
                this::convertRow,
                this::processRowsTurnOn,
                1000);
        YtTable turnOffTable = new YtTable(tableDir + "/turn_off");
        long countTurnOff = ytOperator.readTableSnapshot(turnOffTable,
                tablerow,
                this::convertRow,
                this::processRowsTurnOff,
                1000);
        logger.info("Processed enabling {} for {} clients, disabling for {} clients",
                FEATURE_NAME, countTurnOn, countTurnOff);
        property.set(modTime);
    }

    void processRowsTurnOn(List<ClientId> clientIds) {
        logger.info("Will enable {} for these clients: {}", FEATURE_NAME, clientIds);
        processRows(clientIds, FeatureState.ENABLED);
    }

    void processRowsTurnOff(List<ClientId> clientIds) {
        logger.info("Will disable {} for these clients: {}", FEATURE_NAME, clientIds);
        processRows(clientIds, FeatureState.DISABLED);
    }

    private void processRows(List<ClientId> clientIds, FeatureState state) {
        List<ClientId> existingClientIds = helper.getExistingClientIdsList(clientIds, false);
        if (existingClientIds.isEmpty()) {
            return;
        }
        List<FeatureTextIdToClientIdState> request = existingClientIds.stream()
                .map(i -> new FeatureTextIdToClientIdState().withClientId(i).withTextId(FEATURE_NAME).withState(state))
                .collect(Collectors.toList());
        Result<List<LoginClientIdChiefLoginWithState>> result = service.switchFeaturesStateForClientIds(request);
        if (!result.isSuccessful()) {
            logger.error("Features not updated: {}", result.getErrors());
        }
    }

    private ClientId convertRow(ClientIdTableRow row) {
        return row.getClientId();
    }

    private YPath createDiffTables() throws SQLException, ExecutionException, TimeoutException {
        String tableDir = YtPathUtil.generateTemporaryPath();
        try (YqlConnection connection = (YqlConnection) ytProvider.getYql(YT_CLUSTER, SQLv1).getConnection();
             YqlPreparedStatement statement = (YqlPreparedStatement) connection.prepareStatement(YQL_QUERY)) {
            statement.setLong(1, featureId);
            statement.setString(2, tableDir);
            statement.setString(3, TABLE.getPath());
            ResultSetFuture future = (ResultSetFuture) statement.beginExecuteQuery();
            logger.info("Started YQL execution, OperationID: {}", future.getOperationId());
            future.get(YQL_QUERY_TIMEOUT_MINUTES, MINUTES);
            logger.info("Finished YQL execution, temp dir path: {}", tableDir);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return YPath.simple(tableDir);
    }

    boolean shouldSync(LocalDateTime lastSyncedModTime, YtOperator ytOperator) {
        if (!ytOperator.exists(TABLE)) {
            logger.error("Table for import {} doesn't exists", TABLE);
            setJugglerStatus(JugglerStatus.CRIT, "Table doesn't exists");
            return false;
        }

        if (ytOperator.readTableRowCount(TABLE) < MIN_ROW_COUNT) { // что-то не то с исходными данными
            logger.error("Table {} has fewer than {} rows", TABLE, MIN_ROW_COUNT);
            return false;
        }

        String modificationTimeString = ytOperator.readTableModificationTime(TABLE);
        if (StringUtils.isEmpty(modificationTimeString)) {
            logger.error("Couldn't read the modification_time of table {}", TABLE);
            setJugglerStatus(JugglerStatus.CRIT, "Table modification_time could not be read");
            return false;
        }

        modTime = LocalDateTime.ofInstant(Instant.parse(modificationTimeString), ZoneId.of("UTC"));
        if (modTime.isBefore(LocalDateTime.now().minusDays(YT_TABLE_ACCEPTABLE_AGE_IN_DAYS))) {
            logger.error("Table {} is too old", TABLE);
            setJugglerStatus(JugglerStatus.CRIT, "Table is too old");
            return false;
        }

        if (!modTime.isAfter(lastSyncedModTime)) { // вместо isBefore, чтобы не делать работу при точном совпадении
            logger.info("Feature is already synced");
            return false;
        }

        return true;
    }
}
