package ru.yandex.direct.jobs.feed

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feed.model.UpdateStatus
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.env.TypicalEnvironment
import ru.yandex.direct.jobs.feed.repository.DeleteDuplicatesSiteFeedsRepository
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob
import ru.yandex.direct.utils.mapToSet

@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 150),
    tags = [CheckTag.DIRECT_PRIORITY_2],
    needCheck = NonDevelopmentEnvironment::class,
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_MUNIRA, NotificationRecipient.LOGIN_ALI_AL],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = TypicalEnvironment::class)
class DeleteDuplicatesSiteFeedsJob(
    private val deleteDuplicatesSiteFeedsRepository: DeleteDuplicatesSiteFeedsRepository,
    private val rbacService: RbacService,
    private val feedService: FeedService,
    private val feedRepository: FeedRepository,
    private val grutUacCampaignService: GrutUacCampaignService,
) : DirectShardedJob() {
    override fun execute() {
        val duplicateFeedIds = deleteDuplicatesSiteFeedsRepository.getDuplicateSiteFeedIds(shard)
        val duplicateFeeds = feedRepository.get(shard, duplicateFeedIds)
        val duplicateFeedsByClientIdMap = duplicateFeeds.groupBy { it.clientId }

        val feedIdsToNewFeedIds: Map<Long, Long> = getMainFeeds(duplicateFeedsByClientIdMap)

        deleteDuplicatesSiteFeedsRepository.updateAdGroupPerformanceFeedIds(shard, feedIdsToNewFeedIds)
        deleteDuplicatesSiteFeedsRepository.updateAdGroupDynamicFeedIds(shard, feedIdsToNewFeedIds)

        val feedsToDeleteByClientId = mutableMapOf<Long, List<Feed>>()
        duplicateFeedsByClientIdMap.forEach { (clientId, feeds) ->
            feedsToDeleteByClientId[clientId] = feeds.filter { feedIdsToNewFeedIds.containsKey(it.id) }
        }

        feedsToDeleteByClientId.forEach { (clientId, feeds) ->
            val clientFeedIds = feeds.mapToSet { it.id }
            val clientFeedIdsToNewFeedIds = feedIdsToNewFeedIds.filterKeys { clientFeedIds.contains(it) }
            grutUacCampaignService.updateCampaignFeeds(clientId.toString(), clientFeedIdsToNewFeedIds)
        }

        // Дубликаты фидов могут быть в статусе New, мы хотим их тоже удалять,
        // но в валидации есть проверка статуса, поэтому перед удалением устанавливаем статусы в ERROR
        deleteDuplicatesSiteFeedsRepository.setUpdateStatusError(shard, feedIdsToNewFeedIds.keys)

        feedsToDeleteByClientId.forEach { (clientIdLong, feeds) ->
            val clientId = ClientId.fromLong(clientIdLong)
            val clientUid = rbacService.getChiefByClientId(clientId)
            feedService.deleteFeeds(
                clientId = clientId,
                clientUid = clientUid,
                operatorUid = clientUid,
                feedIds = feeds.map { it.id },
            )
        }
    }

    private fun getMainFeeds(duplicateFeedsByClientId: Map<Long, List<Feed>>): Map<Long, Long> {
        val feedIdToMainFeedId = mutableMapOf<Long, Long>()
        duplicateFeedsByClientId.forEach { (_, duplicateFeeds) ->
            duplicateFeeds.groupBy { it.targetDomain }.forEach { (_, feeds) ->
                val mainFeed = feeds.find {
                    it.offersCount != null && it.offersCount > 0
                } ?: feeds.find {
                    it.updateStatus == UpdateStatus.DONE
                } ?: feeds[0]

                feeds.forEach {
                    if (it.id != mainFeed.id) {
                        feedIdToMainFeedId[it.id] = mainFeed.id
                    }
                }
            }
        }
        return feedIdToMainFeedId
    }
}
