package ru.yandex.direct.jobs.feed

import java.time.Instant
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.ZoneId
import java.time.temporal.ChronoUnit
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.util.RelaxedWorker
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feed.model.FeedSimple
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.service.MbiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.market.client.exception.MarketClientException
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob

private const val CHUNK_UPDATE_SIZE = 1000
private val LOGGER = LoggerFactory.getLogger(DisableFeedsInMBIJob::class.java)

/**
 * Джоба выключающая в MBI фиды, которые были зарегистрированы в MBI, но уже длительное время не используются.
 * Запускается только в продакшне, так как тестовые MBI+DataCamp не готовы к нагрузкам
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 150),
    needCheck = NonDevelopmentEnvironment::class,
    tags = [CheckTag.YT, CheckTag.DIRECT_PRIORITY_2],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_BUHTER],
        method = [NotificationMethod.TELEGRAM], status = [JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = ProductionOnly::class)
class DisableFeedsInMBIJob @Autowired
constructor(
    private val feedRepository: FeedRepository,
    private val mbiService: MbiService,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
    private val rbacService: RbacService,
) : DirectShardedJob() {

    override fun execute() {
        val time = System.currentTimeMillis()
        LOGGER.info("Start on shard: $shard")
        val now = Instant.now().atZone(ZoneId.systemDefault()).toLocalDateTime()
        val lastJobRunTimeProperty: PpcProperty<LocalDateTime> =
            ppcPropertiesSupport.get(PpcPropertyNames.disableFeedsInMBILastTime(shard))

        val lastUsedFromDateTime =
            lastJobRunTimeProperty.getOrDefault(LocalDate.EPOCH.atStartOfDay()).minus(MbiService.FEED_KEEP_MBI_ENABLED_DURATION)
        val lastUsedToDateTime = now.minus(MbiService.FEED_KEEP_MBI_ENABLED_DURATION)

        LOGGER.info(
            "Looking for feeds to disable on shard: $shard in between " +
                "${lastUsedFromDateTime.truncatedTo(ChronoUnit.SECONDS)} " +
                "and ${lastUsedToDateTime.truncatedTo(ChronoUnit.SECONDS)}"
        )

        val feedsToDisable =
            feedRepository.getFeedsSimpleWithSpecifiedLastUsed(shard, lastUsedFromDateTime, lastUsedToDateTime)

        if (feedsToDisable.isEmpty()) {
            LOGGER.info("Got no feeds to disable on shard: $shard")
        } else {
            LOGGER.info("Got ${feedsToDisable.size} feeds to disable on shard: $shard")

            disableFeeds(feedsToDisable, now)
            LOGGER.info("Finish on shard: $shard in ${(System.currentTimeMillis() - time) / 1000} sec")
        }
        lastJobRunTimeProperty.set(now)
    }

    fun disableFeeds(
        feedsToDisable: List<FeedSimple>,
        now: LocalDateTime
    ) {
        val clientIds = feedsToDisable.asSequence()
            .map { ClientId.fromLong(it.clientId) }
            .toSet()

        //Получаем chiefUid'ы для последующих запросов в MBI
        val chiefUidsByClientId = rbacService.getChiefsByClientIds(clientIds)
        val relaxedWorker = RelaxedWorker()

        //Почанково регистрируем фиды и записываем результаты в базу
        feedsToDisable
            .chunked(CHUNK_UPDATE_SIZE)
            .forEachIndexed { index, feedListChunk ->
                LOGGER.info(
                    "Disabling ${index + 1} from ${feedsToDisable.size / CHUNK_UPDATE_SIZE + 1} " +
                        "chunk feeds to MBI on shard: $shard"
                )
                val skippedFeeds = mutableListOf<FeedSimple>()
                feedListChunk.forEach { feed ->
                    val clientId = ClientId.fromLong(feed.clientId)
                    val chiefUid = chiefUidsByClientId[clientId]!!
                    try {
                        relaxedWorker.callAndRelax<Unit, RuntimeException> {
                            mbiService.sendFeed(clientId, chiefUid, feed)
                        }
                    } catch (e: MarketClientException) {
                        LOGGER.warn(
                            "Got market client exception during sending feed with id: ${feed.id}. " +
                                "Message: ${e.localizedMessage}"
                        )
                        skippedFeeds.add(feed)
                    }
                }
                LOGGER.info(
                    "Updating last_used for skipped feeds: ${skippedFeeds.size} "
                )
                updateLastUsed(shard, skippedFeeds, now)
                LOGGER.info(
                    "Done with ${index + 1} from ${feedsToDisable.size / CHUNK_UPDATE_SIZE + 1} " +
                        "chunk on shard: $shard"
                )
            }
    }

    private fun updateLastUsed(shard: Int, skippedFeeds: List<FeedSimple>, now: LocalDateTime) {
        val appliedChanges = skippedFeeds.map {
            ModelChanges(it.id, Feed::class.java)
                .process(now.plusSeconds(1L), Feed.LAST_USED)
                .applyTo(it as Feed)
        }
        feedRepository.update(shard, appliedChanges)
    }
}
