package ru.yandex.direct.jobs.feed

import java.util.EnumSet
import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.core.entity.feed.converter.FeedConverter.extractBusinessIdAndShopId
import ru.yandex.direct.core.entity.feed.feedWithUsageTypeToAppliedChanges
import ru.yandex.direct.core.entity.feed.model.FeedUsageType
import ru.yandex.direct.core.entity.feed.repository.BlrtYtRepository
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.dbutil.model.BusinessIdAndShopId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.env.NonProductionEnvironment
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob

/**
 * Джоба, синхронизирующая используемость фидов с заданиями на генерацию в BannerLand.
 * Используемыми считаются те фиды, что прокачаны в MBI/DataCamp (т.е. для них есть BusinessID и ShopID)
 * и по которым есть с соответствующими ID активные задания на генерацию в BLRT.
 * Все остальные фиды считаем неиспользуемыми. Используемые фиды затем включаются в MBI (FeedUsageTypeProcessor),
 * а неиспользуемые через некоторое время выключаются (DisableFeedsInMBIJob), чтобы не создавать лишней нагрузки.
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 30),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.YT, CheckTag.DIRECT_PRIORITY_1],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_ALI_AL],
        method = [NotificationMethod.TELEGRAM], status = [JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 5 * 60, needSchedule = ProductionOnly::class)
@JugglerCheck(
    ttl = JugglerCheck.Duration(hours = 3),
    needCheck = NonProductionEnvironment::class,
    tags = [CheckTag.YT, CheckTag.DIRECT_PRIORITY_2]
)
@Hourglass(periodInSeconds = 30 * 60, needSchedule = NonProductionEnvironment::class)
class RecalculateFeedsUsageTypesJob(
    private val blrtYtRepository: BlrtYtRepository,
    private val feedRepository: FeedRepository,
    private val shardHelper: ShardHelper
) : DirectJob() {
    private val logger = LoggerFactory.getLogger(RecalculateFeedsUsageTypesJob::class.java)

    override fun execute() {
        logger.info("START")

        val businessIdsAndShopIds = emptySequence<BusinessIdAndShopId>()
            .plus(blrtYtRepository.getDynamicTasksBusinessIdsAndShopIds())
            .plus(blrtYtRepository.getPerformanceTasksBusinessIdsAndShopIds())
            .toSet()
        logger.info("${businessIdsAndShopIds.size} feeds are in use")

        shardHelper.forEachShard { shard ->
            logger.info("Processing shard $shard")

            val feeds = feedRepository.getAllDataCampFeedsSimple(shard)
            logger.info("${feeds.size} feeds total")

            val appliedChanges = feedWithUsageTypeToAppliedChanges(feeds.associateWith {
                if (businessIdsAndShopIds.contains(extractBusinessIdAndShopId(it))) {
                    EnumSet.of(FeedUsageType.GOODS_ADS)
                } else {
                    EnumSet.noneOf(FeedUsageType::class.java)
                }
            })
            val (feedsNotInUse, feedsInUse) = appliedChanges.map { it.model }.partition { it.usageTypes.isEmpty() }
            logger.info("${appliedChanges.size} feeds with usage types changes")
            logger.info("${feedsNotInUse.size} feeds are no longer in use: ${feedsNotInUse.map { it.id }}")
            logger.info("${feedsInUse.size} feeds are now in use: ${feedsInUse.map { it.id }}")

            feedRepository.update(shard, appliedChanges)
        }

        logger.info("FINISH")
    }
}
