package ru.yandex.direct.jobs.feed

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.util.RelaxedWorker
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.core.entity.feed.service.MbiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.market.client.exception.MarketClientException
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob

private const val CHUNK_UPDATE_SIZE = 1000
private val LOGGER = LoggerFactory.getLogger(SendFeedToMBIJob::class.java)

/**
 * Джоба регистрирующая фиды, которые еще не были зарегистрированы, и которые могут быть зарегистрированы в MBI.
 * Запускается только в продакшне, так как тестовые MBI+DataCamp не готовы к нагрузкам
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 20),
    needCheck = NonDevelopmentEnvironment::class,
    tags = [CheckTag.YT, CheckTag.DIRECT_PRIORITY_0],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_BUHTER],
        method = [NotificationMethod.TELEGRAM], status = [JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 60, needSchedule = ProductionOnly::class)
class SendFeedToMBIJob @Autowired constructor(
    private val feedService: FeedService,
    private val rbacService: RbacService,
    private val feedRepository: FeedRepository,
    private val mbiService: MbiService
) : DirectShardedJob() {
    override fun execute() {
        val time = System.currentTimeMillis()
        LOGGER.info("Start on shard: $shard")
        val feedsToSend = feedService.getFeedsToSendToMBI(shard).associateBy { it.id }
        if (feedsToSend.isEmpty()) {
            LOGGER.info("Got no feeds to send on shard: $shard")
        } else {
            LOGGER.info("Got ${feedsToSend.size} feeds to send on shard: $shard")

            val clientIds = feedsToSend.values
                .asSequence()
                .map { ClientId.fromLong(it.clientId) }
                .toSet()
            // Получаем chiefUid'ы для последующих запросов в MBI
            val chiefUidsByClientId = rbacService.getChiefsByClientIds(clientIds)

            val relaxedWorker = RelaxedWorker()
            // Почанково регистрируем фиды и записываем результаты в базу
            feedsToSend.values
                .asSequence()
                .chunked(CHUNK_UPDATE_SIZE)
                .forEachIndexed { index, feedListChunk ->
                    LOGGER.info(
                        "Sending ${index + 1} from ${feedsToSend.size / CHUNK_UPDATE_SIZE + 1} " +
                            "chunk feeds to MBI on shard: $shard"
                    )
                    feedListChunk
                        .forEach { feed ->
                            // Получаем маркетные id для фидов (пока по одному)
                            val clientId = ClientId.fromLong(feed.clientId)
                            try {
                                relaxedWorker.callAndRelax<MbiService.SendFeedResult, RuntimeException> {
                                    mbiService.sendFeed(clientId, chiefUidsByClientId[clientId]!!, feed)
                                }
                            } catch (e: MarketClientException) {
                                LOGGER.warn(
                                    "Got market client exception during sending feed with id: ${feed.id}. " +
                                        "Message: ${e.localizedMessage}"
                                )
                            }
                        }
                    LOGGER.info(
                        "Updating ${index + 1} from ${feedsToSend.size / CHUNK_UPDATE_SIZE + 1} " +
                            "chunk feeds in DB on shard: $shard"
                    )
                    LOGGER.info(
                        "Done with ${index + 1} from ${feedsToSend.size / CHUNK_UPDATE_SIZE + 1} " +
                            "chunk on shard: $shard"
                    )
                }
        }
        LOGGER.info("Finish on shard: $shard in ${(System.currentTimeMillis() - time) / 1000} sec")
    }
}
