package ru.yandex.direct.jobs.feed

import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.bmapi.client.BmapiClient
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.DEFAULT_FEED_TO_BANNERLAND_ITERATION_CAPACITY
import ru.yandex.direct.core.entity.client.service.ClientLimitsService
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.jobs.feed.service.SendFeedsToBannerLandJobService
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob
import java.time.Duration
import java.time.temporal.ChronoUnit

private val LOGGER = LoggerFactory.getLogger(SendNewFeedsToBannerLandJob::class.java)
private const val DEFAULT_ITERATION_CAPACITY = 5

/**
 * Джоба отправляющая фиды с update_status New в BannerLand (BmAPI)
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(hours = 2),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.YT, CheckTag.DIRECT_PRIORITY_0],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_BUHTER],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT]
    )]
)
@Hourglass(periodInSeconds = 60, needSchedule = ProductionOnly::class)
class SendNewFeedsToBannerLandJob(
    private val feedService: FeedService,
    private val clientLimitsService: ClientLimitsService,
    private val ppcPropertySupport: PpcPropertiesSupport,
    feedRepository: FeedRepository,
    bmapiClient: BmapiClient
) : DirectShardedJob() {

    private val sendFeedsToBannerLandJobService: SendFeedsToBannerLandJobService =
        SendFeedsToBannerLandJobService(feedService, feedRepository, bmapiClient, LOGGER)

    override fun execute() {
        LOGGER.info("START on shard $shard")
        val iterationCapacity =
            ppcPropertySupport.get(DEFAULT_FEED_TO_BANNERLAND_ITERATION_CAPACITY, Duration.of(600, ChronoUnit.SECONDS))
                .getOrDefault(DEFAULT_ITERATION_CAPACITY)
        val feedsToSendToBmAPI = feedService.getNewFeedsToSendToBmAPI(shard, iterationCapacity)
        if (feedsToSendToBmAPI.isEmpty()) {
            LOGGER.info("No new feeds to send to BannerLand on shard $shard")
            return
        }
        LOGGER.info("Got ${feedsToSendToBmAPI.size} new feeds to send to BannerLand on shard $shard")
        val clientIds = feedsToSendToBmAPI.map { ClientId.fromLong(it.clientId) }
        val maxFileSizeByClientId = clientLimitsService.massGetClientLimits(clientIds)
            .associate { it.clientId.asLong() to it.feedsMaxFileSizeOrDefault }

        sendFeedsToBannerLandJobService.sendFeedsToBmApi(shard, feedsToSendToBmAPI, maxFileSizeByClientId)
        LOGGER.info("FINISH on shard $shard")
    }
}
