package ru.yandex.direct.jobs.feed.repository

import org.jooq.Field
import org.jooq.impl.DSL
import org.jooq.impl.DSL.groupConcat
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.StatusBsSynced
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbschema.ppc.enums.FeedsSource
import ru.yandex.direct.dbschema.ppc.enums.FeedsUpdateStatus
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.JooqMapperUtils

@Repository
class DeleteDuplicatesSiteFeedsRepository(
    private val dslContextProvider: DslContextProvider,
    private val adGroupRepository: AdGroupRepository,
) {
    fun getDuplicateSiteFeedIds(shard: Int): List<Long> {
        val concatenatedIds: Field<String> = groupConcat(Tables.FEEDS.FEED_ID).separator(",")
            .`as`(DSL.name("concatenatedIds"))
        val concatenatedDuplicateIds = dslContextProvider.ppc(shard)
            .select(Tables.FEEDS.TARGET_DOMAIN, Tables.FEEDS.CLIENT_ID, concatenatedIds)
            .from(Tables.FEEDS)
            .where(Tables.FEEDS.SOURCE.eq(FeedsSource.site)).and(Tables.FEEDS.TARGET_DOMAIN.isNotNull)
            .groupBy(Tables.FEEDS.TARGET_DOMAIN, Tables.FEEDS.CLIENT_ID)
            .having(DSL.count().greaterThan(1))
            .fetch(concatenatedIds)
        return concatenatedDuplicateIds.joinToString(",").split(",").mapNotNull { it.toLongOrNull() }
    }

    fun updateAdGroupPerformanceFeedIds(
        shard: Int,
        feedIdsToNewFeedIds: Map<Long, Long>
    ) {
        val feedIds = feedIdsToNewFeedIds.keys
        val adgroupPerformanceIdsToFeedIds = dslContextProvider.ppc(shard)
            .select(Tables.ADGROUPS_PERFORMANCE.PID, Tables.FEEDS.FEED_ID)
            .from(Tables.FEEDS)
            .join(Tables.ADGROUPS_PERFORMANCE).on(Tables.ADGROUPS_PERFORMANCE.FEED_ID.eq(Tables.FEEDS.FEED_ID))
            .where(Tables.FEEDS.FEED_ID.`in`(feedIds))
            .fetchMap(Tables.ADGROUPS_PERFORMANCE.PID, Tables.FEEDS.FEED_ID)

        val adgroupPerformanceIdsToFeedIdsToNewFeedIds = adgroupPerformanceIdsToFeedIds.mapValues { feedIdsToNewFeedIds[it.value] }

        val feedIdCase = JooqMapperUtils.makeCaseStatement(
            Tables.ADGROUPS_PERFORMANCE.PID,
            Tables.ADGROUPS_PERFORMANCE.FEED_ID,
            adgroupPerformanceIdsToFeedIdsToNewFeedIds
        )

        dslContextProvider.ppc(shard)
            .update(Tables.ADGROUPS_PERFORMANCE)
            .set(Tables.ADGROUPS_PERFORMANCE.FEED_ID, feedIdCase)
            .where(Tables.ADGROUPS_PERFORMANCE.PID.`in`(adgroupPerformanceIdsToFeedIdsToNewFeedIds.keys))
            .execute()

        adGroupRepository.updateStatusBsSynced(shard, adgroupPerformanceIdsToFeedIdsToNewFeedIds.keys, StatusBsSynced.NO)
    }

    fun updateAdGroupDynamicFeedIds(
        shard: Int,
        feedIdsToNewFeedIds: Map<Long, Long>
    ) {
        val feedIds = feedIdsToNewFeedIds.keys
        val adgroupDynamicIdsToFeedIds = dslContextProvider.ppc(shard)
            .select(Tables.ADGROUPS_DYNAMIC.PID, Tables.FEEDS.FEED_ID)
            .from(Tables.FEEDS)
            .join(Tables.ADGROUPS_DYNAMIC).on(Tables.ADGROUPS_DYNAMIC.FEED_ID.eq(Tables.FEEDS.FEED_ID))
            .where(Tables.FEEDS.FEED_ID.`in`(feedIds))
            .fetchMap(Tables.ADGROUPS_DYNAMIC.PID, Tables.FEEDS.FEED_ID)

        val adgroupDynamicIdsToNewFeedIds = adgroupDynamicIdsToFeedIds.mapValues { feedIdsToNewFeedIds[it.value] }

        val feedIdCase = JooqMapperUtils.makeCaseStatement(
            Tables.ADGROUPS_DYNAMIC.PID,
            Tables.ADGROUPS_DYNAMIC.FEED_ID,
            adgroupDynamicIdsToNewFeedIds
        )

        dslContextProvider.ppc(shard)
            .update(Tables.ADGROUPS_DYNAMIC)
            .set(Tables.ADGROUPS_DYNAMIC.FEED_ID, feedIdCase)
            .where(Tables.ADGROUPS_DYNAMIC.PID.`in`(adgroupDynamicIdsToNewFeedIds.keys))
            .execute()

        adGroupRepository.updateStatusBsSynced(shard, adgroupDynamicIdsToFeedIds.keys, StatusBsSynced.NO)
    }

    fun setUpdateStatusError(shard: Int, feedIds: Collection<Long>) {
        dslContextProvider.ppc(shard)
            .update(Tables.FEEDS)
            .set(Tables.FEEDS.UPDATE_STATUS, FeedsUpdateStatus.Error)
            .where(Tables.FEEDS.FEED_ID.`in`(feedIds))
            .execute()
    }
}
