package ru.yandex.direct.jobs.feed.service

import org.slf4j.Logger
import org.springframework.stereotype.Service
import ru.yandex.direct.asynchttp.ErrorResponseWrapperException
import ru.yandex.direct.bmapi.client.BmapiClient
import ru.yandex.direct.bmapi.client.model.BmApiFeedInfoResponse
import ru.yandex.direct.bmapi.client.model.BmApiRequest
import ru.yandex.direct.core.entity.feed.model.FeedSimple
import ru.yandex.direct.core.entity.feed.model.UpdateStatus
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.service.FeedService

class SendFeedsToBannerLandJobService(
    private val feedService: FeedService,
    private val feedRepository: FeedRepository,
    private val bmapiClient: BmapiClient,
    private val logger: Logger
) {

    fun sendFeedsToBmApi(
        shard: Int,
        feedsToSendToBmAPI: List<FeedSimple>,
        maxFileSizeByClientId: Map<Long, Long>,
    ) {
        val requests: List<BmApiRequest> = feedsToSendToBmAPI.map {
            BmApiRequest(
                it.url,
                it.businessType,
                it.feedType,
                maxFileSizeByClientId[it.clientId]!!,
                it.updateStatus,
                it.id,
                it.login,
                it.plainPassword
            )
        }
        feedRepository.setFeedsStatus(shard, UpdateStatus.UPDATING, feedsToSendToBmAPI)
        val results = bmapiClient.processFeeds(requests)
        val successResultsByFeedId = HashMap<Long, BmApiFeedInfoResponse>()
        results.forEach { (feedId, result) ->
            if (result.success == null && !result.errors.isNullOrEmpty()) {
                logger.error(
                    "Got exception while sending feed $feedId toBmAPI: " +
                            result.errors.map { throwable ->
                                throwable.toString() +
                                        if (throwable is ErrorResponseWrapperException && throwable.response != null) {
                                            throwable.response.toString()
                                        } else {
                                            ""
                                        }
                            }
                )
            } else {
                logger.info("Feed with id $feedId processed by BmAPI")
                successResultsByFeedId[feedId] = result.success
            }
        }
        feedService.saveFeedFromBmApiResponse(shard, successResultsByFeedId)
    }
}
