package ru.yandex.direct.jobs.fetchmobilecontentfromstore;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.function.Function;
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentQueueItem;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentFetchQueueRepository;
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.yt.ytclient.proxy.request.CreateNode;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_API_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Собирает мобильные приложения из очереди (таблица ppcdict.mobile_content_fetch_queue) и делает запрос
 * в Аврору (=запись в Yt таблицу) на их прокачку из стора.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = ProductionOnly.class,

        //PRIORITY: Временно поставили приоритет по умолчанию;
        tags = {DIRECT_PRIORITY_1_NOT_READY, DIRECT_API_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_API_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = NonProductionEnvironment.class,

        //PRIORITY: Временно поставили приоритет по умолчанию;
        tags = {DIRECT_PRIORITY_1_NOT_READY, DIRECT_API_TEAM, JOBS_RELEASE_REGRESSION}
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class FetchMobileContentFromStoreJob extends DirectJob {

    private final Logger logger = LoggerFactory.getLogger(FetchMobileContentFromStoreJob.class);

    static final TableSchema SCHEMA = new TableSchema.Builder()
            .addValue("url", ColumnValueType.STRING)
            .setUniqueKeys(false)
            .build();

    private static final Function<Iterator<Record>, Set<String>> MAPPER = it -> Cf.x(it).stream()
            .map(Record::getUrl)
            .collect(Collectors.toSet());

    private final MobileContentFetchQueueRepository mobileContentFetchQueueRepository;
    private final FetchMobileContentFromStoreJobConfig config;
    private final YtProvider ytProvider;

    @Autowired
    public FetchMobileContentFromStoreJob(MobileContentFetchQueueRepository mobileContentFetchQueueRepository,
                                          YtProvider ytProvider, DirectConfig directConfig) {
        this.ytProvider = ytProvider;
        this.mobileContentFetchQueueRepository = mobileContentFetchQueueRepository;

        config = new FetchMobileContentFromStoreJobConfig(directConfig);
    }

    @Override
    public void execute() {
        List<MobileContentQueueItem> queueItems = mobileContentFetchQueueRepository.getAllItems();

        if (queueItems.isEmpty()) {
            logger.info("Queue is empty");
            return;
        }

        logger.info("Queue contains {} elements", queueItems.size());

        deleteUnusedItemsFromQueue(queueItems);
        Map<OsType, List<MobileContentQueueItem>> queueItemsByOsType = groupQueueItemsByOsType(queueItems);

        config.getYtTablePathsByOsTypes().forEach((osType, ytTablePath) -> {
            List<MobileContentQueueItem> items = queueItemsByOsType.get(osType);
            if (items == null || items.isEmpty()) {
                return;
            }

            boolean success = writeToYt(ytTablePath, items);

            if (success) {
                mobileContentFetchQueueRepository.deleteItems(items);
            }

            logger.info("Write for OsType {} is successful: {}", osType.name(), success);
        });
    }

    /**
     * Создает новую таблицу в Yt и записывает в нее заданные элементы.
     *
     * @param items элементы для записи
     * @return true, если элементы очереди успешно записаны в Yt и удалены из очереди, иначе false
     */
    boolean writeToYt(YtTablePath ytTablePath, List<MobileContentQueueItem> items) {
        Yt yt = ytProvider.get(ytTablePath.getYtCluster());
        String ytTableDir = ytTablePath.getYtTableDir();

        String tableName = new SimpleDateFormat("yyyy.MM.dd-HH:mm").format(new Date());
        YPath yTablePath = YPath.simple(YtPathUtil.generatePath(ytTableDir, tableName));

        // Таблица ppcdict.mobile_content_fetch_queue и MobileContentFetchQueueRepository не поддерживают
        // уникальность url, поэтому делаем это сами с помощью Set
        Set<Record> yUrlModels = StreamEx.of(items)
                .map(item -> new Record(item.getUrl()))
                .toSet();
        var createNode = new CreateNode(yTablePath, CypressNodeType.TABLE, Cf.map(YtUtils.SCHEMA_ATTR, SCHEMA.toYTree()));
        createNode.setRecursive(true);
        yt.cypress().create(createNode);
        yt.tables().write(yTablePath, Record.YT_TYPE, Cf.wrap(yUrlModels).iterator());

        return yt.cypress().exists(yTablePath) && !yt.tables().read(yTablePath, Record.YT_TYPE, MAPPER).isEmpty();
    }

    /**
     * Группирует элементы очереди по типу ОС.
     *
     * @param queueItems все элементы очереди
     * @return элементы очереди, сгруппированные по типу ОС
     */
    Map<OsType, List<MobileContentQueueItem>> groupQueueItemsByOsType(List<MobileContentQueueItem> queueItems) {
        return StreamEx.of(queueItems)
                .mapToEntry(item -> MobileAppStoreUrlParser.parse(item.getUrl()), item -> item)
                .filterKeys(Optional::isPresent)
                .mapKeys(k -> k.get().getOsType())
                .grouping();
    }

    /**
     * Удаляет из очереди элементы, для которых невозможно сделать запрос (неизвестен тип ОС или путь к директории в Yt).
     *
     * @param queueItems все элементы очереди
     */
    void deleteUnusedItemsFromQueue(List<MobileContentQueueItem> queueItems) {
        mobileContentFetchQueueRepository.deleteItems(StreamEx.of(queueItems)
                .mapToEntry(item -> MobileAppStoreUrlParser.parse(item.getUrl()), item -> item)
                .filterKeys(k -> !k.isPresent() || !config.getYtTablePathsByOsTypes().containsKey(k.get().getOsType()))
                .values()
                .toList());
    }
}
