package ru.yandex.direct.jobs.filterdomains;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Выгружает все записи из таблицы ppc.filter_domains в таблицу в YT hahn.[//home/direct/export/filter_domains]
 * <p>
 * Выгрузка выполняется раз в 4 часа.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 8),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2, CheckTag.YT}
)
@Hourglass(cronExpression = "0 0 */4 ? * *", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class ExportFilterDomainsToYtJob extends DirectJob {
    private static final YtCluster EXPORT_CLUSTER = YtCluster.HAHN;

    private final ShardHelper shardHelper;
    private final YtProvider ytProvider;
    private final DomainService domainService;

    private static final Logger logger = LoggerFactory.getLogger(FilterDomainsYtUploader.class);

    @Autowired
    public ExportFilterDomainsToYtJob(YtProvider ytProvider, DomainService domainService, ShardHelper shardHelper) {
        this.shardHelper = shardHelper;
        this.ytProvider = ytProvider;
        this.domainService = domainService;
    }

    @Override
    public void execute() {
        logger.info("ExportFilterDomainsToYtJob started");

        FilterDomainsYtUploader uploader = new FilterDomainsYtUploader(ytProvider, EXPORT_CLUSTER, domainService);

        uploader.createTableIfAbsent();

        shardHelper.dbShards().forEach(uploader::export);

        logger.info("ExportFilterDomainsToYtJob finished");
    }
}
