package ru.yandex.direct.jobs.filterdomains;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.common.util.AutoCloseableIterator;
import ru.yandex.direct.core.entity.domain.model.FilterDomain;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static java.util.Arrays.asList;
import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;
import static ru.yandex.direct.ytwrapper.YtPathUtil.generatePath;

@ParametersAreNonnullByDefault
public class FilterDomainsYtUploader {
    private static final Duration OPERATION_TIMEOUT = Duration.ofSeconds(60);
    private static final String FILTER_DOMAIN_EXPORT_PATH = "export/filter_domains";
    private static final String DOMAIN = "domain";
    private static final String FILTER_DOMAIN = "filter_domain";

    private static final TableSchema SCHEMA = new TableSchema.Builder()
            .addKey(DOMAIN, ColumnValueType.STRING)
            .addValue(FILTER_DOMAIN, ColumnValueType.STRING)
            .build();

    private static final int CHUNK_SIZE = 5000;

    private static final Logger logger = LoggerFactory.getLogger(FilterDomainsYtUploader.class);

    private final String path;
    private final DomainService domainService;
    private final YtProvider ytProvider;
    private final YtCluster exportCluster;

    public FilterDomainsYtUploader(YtProvider ytProvider, YtCluster exportCluster, DomainService domainService) {
        this.path = generatePath(ytProvider.getClusterConfig(exportCluster).getHome(), relativePart(),
                FILTER_DOMAIN_EXPORT_PATH);
        this.domainService = domainService;
        this.ytProvider = ytProvider;
        this.exportCluster = exportCluster;
    }

    /**
     * Экспорт данных для шарда.
     *
     * @param shard Номер шарда.
     */
    public void export(int shard) {
        logger.info("Start exporting filter_domains from shard {} to YT {}.", shard, path);

        int recordCount = 0;

        YtDynamicOperator dynamicOperator = ytProvider.getDynamicOperator(exportCluster);

        ApiServiceTransactionOptions txOpts = new ApiServiceTransactionOptions(ETransactionType.TT_TABLET)
                .setSticky(true)
                .setPing(true)
                .setTimeout(OPERATION_TIMEOUT);

        try (AutoCloseableIterator<List<FilterDomain>> domains = domainService.allFilterDomains(shard, CHUNK_SIZE)) {
            while (domains.hasNext()) {
                List<List<String>> data = new ArrayList<>(CHUNK_SIZE);

                try (TraceProfile profile = Trace.current().profile("jobs:export_filter_domains:read_chunk")) {
                    domains.next().forEach(rec -> data.add(asList(rec.getDomain(), rec.getFilterDomain())));
                }

                recordCount += data.size();

                try (TraceProfile profile = Trace.current().profile("jobs:export_filter_domains:write_chunk")) {
                    dynamicOperator.runInTransaction(tx ->
                            tx.modifyRows(new ModifyRowsRequest(path, SCHEMA)
                                    .setRequireSyncReplica(false)
                                    .addUpdates(data, false)
                            ).join(), txOpts); // IGNORE-BAD-JOIN DIRECT-149116
                }

                logger.debug("Exported {} records from shard {} to YT {}", recordCount, shard, path);
            }
        }

        logger.debug("Finish exporting filter_domains from shard {} to YT. {} records exported", shard, recordCount);
    }

    /**
     * Создаёт таблицу в YT, если она не существует.
     */
    void createTableIfAbsent() {
        YtOperator operator = ytProvider.getOperator(exportCluster);
        Yt yt = operator.getYt();
        YPath table = YPath.simple(path);
        logger.debug("Cluster: '{}', Table: '{}'", exportCluster, table);

        if (yt.cypress().exists(table)) {
            logger.debug("Table '{}' already exists", table);

            return;
        }

        try {
            logger.info("Creating table '{}'", table);

            yt.cypress().create(table, CypressNodeType.TABLE, true, false,
                    Cf.map(YtUtils.SCHEMA_ATTR, SCHEMA.toYTree(), "dynamic", YTree.booleanNode(true)));

            logger.info("Mounting table '{}'", table);
            operator.mount(table, 60_000);
            logger.info("Successfully mounted table '{}'", table);
        } catch (Exception e) {
            try {
                if (yt.cypress().exists(table)) {
                    yt.cypress().remove(table);
                }
            } catch (Exception e1) {
                logger.error("Error in cleanup", e1);
            }

            throw e;
        }
    }
}
