package ru.yandex.direct.jobs.freelancers;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.freelancer.container.FreelancerProjectQueryContainer;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProject;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProjectStatus;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerProjectRepository;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.jobs.configuration.DirectExportYtClustersParametersSource;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter.enabledFreelancers;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.ENABLE_EXPORT_FREELANCERS_PROJECTS_TO_YT;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Выгружает записи из таблицы ppc.freelancers_projects в таблицу в YT hahn.[//home/direct/export/freelancers/freelancers_projects]
 * <p>
 * Выгрузка выполняется раз в 15 минут
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2),
        needCheck = NonDevelopmentEnvironment.class,
        tags = {DIRECT_PRIORITY_1, JOBS_RELEASE_REGRESSION},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_MAXLOG,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 15 * 60, needSchedule = NonDevelopmentEnvironment.class)
@ParameterizedBy(parametersSource = DirectExportYtClustersParametersSource.class)
@ParametersAreNonnullByDefault
public class ExportFreelancersProjectsToYtJob extends DirectParameterizedJob<YtCluster> {
    private static final int FREELANCERS_CHUNK_SIZE = 50;
    private static final RelaxedWorker relaxedWorker = new RelaxedWorker(2.0);

    private static final Logger logger = LoggerFactory.getLogger(ExportFreelancersProjectsToYtJob.class);

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final DirectExportYtClustersParametersSource parametersSource;
    private final YtProvider ytProvider;
    private final ShardHelper shardHelper;
    private final FreelancerProjectRepository freelancerProjectRepository;
    private final FreelancerService freelancerService;
    private final UserService userService;


    @Autowired
    public ExportFreelancersProjectsToYtJob(
            PpcPropertiesSupport ppcPropertiesSupport,
            DirectExportYtClustersParametersSource parametersSource,
            YtProvider ytProvider,
            ShardHelper shardHelper,
            FreelancerProjectRepository freelancerProjectRepository,
            FreelancerService freelancerService,
            UserService userService) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.parametersSource = parametersSource;
        this.ytProvider = ytProvider;
        this.shardHelper = shardHelper;
        this.freelancerProjectRepository = freelancerProjectRepository;
        this.freelancerService = freelancerService;
        this.userService = userService;
    }

    @Override
    public void execute() {
        if (!isJobEnabled()) {
            logger.info("Skip processing. Job is not enabled.");
            return;
        }

        YtCluster ytCluster = parametersSource.convertStringToParam(getParam());
        FreelancersProjectsYtUploader uploader = new FreelancersProjectsYtUploader(ytProvider, ytCluster);
        uploader.createTableIfAbsent();

        Set<Long> allFreelancerIds = getAllFreelancersIds();
        for (List<Long> chunk : Iterables.partition(allFreelancerIds, FREELANCERS_CHUNK_SIZE)) {
            relaxedWorker.runAndRelax(() -> {
                List<FreelancerProject> freelancersProjects = getFreelancersProjects(chunk);
                List<YtFreelancerProject> ytFreelancersProjects = convertToYt(freelancersProjects);
                uploader.upload(ytFreelancersProjects);
            });
        }
    }

    boolean isJobEnabled() {
        String propValue = ppcPropertiesSupport.get(ENABLE_EXPORT_FREELANCERS_PROJECTS_TO_YT.getName());
        return propValue != null && propValue.trim().equals("1");
    }

    private Set<Long> getAllFreelancersIds() {
        List<Freelancer> freelancers = freelancerService.getFreelancers(enabledFreelancers().build());
        return listToSet(freelancers, FreelancerBase::getFreelancerId);
    }

    List<FreelancerProject> getFreelancersProjects(List<Long> freelancerIds) {
        List<FreelancerProject> result = new ArrayList<>();
        shardHelper.forEachShard(shard -> {
            List<FreelancerProject> projects = getFreelancersProjectsInShard(shard, freelancerIds);
            result.addAll(projects);
        });
        return result;
    }

    private List<FreelancerProject> getFreelancersProjectsInShard(int shard, List<Long> freelancerIds) {
        FreelancerProjectQueryContainer queryContainer = FreelancerProjectQueryContainer.builder()
                .withFreelancerIds(freelancerIds)
                .withStatuses(
                        FreelancerProjectStatus.INPROGRESS,
                        FreelancerProjectStatus.CANCELLEDBYCLIENT,
                        FreelancerProjectStatus.CANCELLEDBYFREELANCER)
                .withStarted(true)
                .build();
        return freelancerProjectRepository.get(shard, queryContainer, LimitOffset.maxLimited());
    }

    /**
     * Преобразует проекты в Yt-представление,
     * данные проектов дополняются данными фрилансера (login, name)
     */
    List<YtFreelancerProject> convertToYt(List<FreelancerProject> freelancersProjects) {
        Set<Long> freelancerIds = listToSet(freelancersProjects, FreelancerProject::getFreelancerId);

        List<Freelancer> freelancers = freelancerService.getFreelancers(freelancerIds);
        Map<Long, Freelancer> freelancersByIds = listToMap(freelancers, FreelancerBase::getFreelancerId);

        Map<ClientId, String> loginsByClientIds =
                userService.getChiefsLoginsByClientIds(mapList(freelancerIds, ClientId::fromLong));

        Map<String, Long> uidsByLogins = userService.massGetUidByLogin(new ArrayList<>(loginsByClientIds.values()));

        return StreamEx.of(freelancersProjects)
                .map(fp -> convertRow(fp, freelancersByIds, loginsByClientIds, uidsByLogins))
                .nonNull()
                .toList();
    }

    private YtFreelancerProject convertRow(FreelancerProject project, Map<Long, Freelancer> freelancersByIds,
                                           Map<ClientId, String> loginsByClientIds, Map<String, Long> uidsByLogins) {
        checkNotNull(project, "project can not be null");
        checkNotNull(project.getId(), "id is not set for project %s", project);
        checkNotNull(project.getFreelancerId(), "freelancerId is not set for project %s", project);

        Freelancer freelancer = freelancersByIds.get(project.getFreelancerId());
        String login = loginsByClientIds.get(ClientId.fromLong(project.getFreelancerId()));
        Long uid = uidsByLogins.get(login);
        // для активных сотрудничеств ставим finishDate = null, для завершившихся он совпадает с updatedTime
        LocalDateTime finishDate =
                FreelancerProjectStatus.INPROGRESS.equals(project.getStatus()) ? null : project.getUpdatedTime();

        if (freelancer != null && login != null) {
            return new YtFreelancerProject()
                    .withProjectId(project.getId())
                    .withFreelancerClientId(project.getFreelancerId())
                    .withCustomerClientId(project.getClientId())
                    .withStartDate(project.getStartedTime())
                    .withFinishDate(finishDate)
                    .withFreelancerName(freelancer.getFirstName() + " " + freelancer.getSecondName())
                    .withFreelancerUid(uid)
                    .withFreelancerLogin(login);
        } else {
            return null;
        }
    }
}
