package ru.yandex.direct.jobs.freelancers;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static ru.yandex.direct.jobs.util.yt.YtEnvPath.relativePart;

/**
 * Класс выполняющий непосредственную загрузку выбранных данных в динамическую таблицу в YT
 */
@ParametersAreNonnullByDefault
class FreelancersProjectsYtUploader {
    private static final String FREELANCERS_PROJECTS_EXPORT_PATH = "export/freelancers/freelancers_projects";

    private static final DateTimeFormatter DATE_TIME_FORMATTER =
            DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    private static final String PROJECT_ID_COLUMN = "project_id";
    private static final String FREELANCER_CLIENT_ID_COLUMN = "freelancer_client_id";
    private static final String CUSTOMER_CLIENT_ID_COLUMN = "customer_client_id";
    private static final String START_DATE_COLUMN = "start_date";
    private static final String FINISH_DATE_COLUMN = "finish_date";
    private static final String FREELANCER_LOGIN_COLUMN = "freelancer_login";
    private static final String FREELANCER_LOGIN_UID = "freelancer_uid";
    private static final String FREELANCER_NAME_COLUMN = "freelancer_name";

    private static final TableSchema schema = new TableSchema.Builder()
            .addKey(PROJECT_ID_COLUMN, ColumnValueType.INT64)
            .addValue(FREELANCER_CLIENT_ID_COLUMN, ColumnValueType.INT64)
            .addValue(CUSTOMER_CLIENT_ID_COLUMN, ColumnValueType.INT64)
            .addValue(START_DATE_COLUMN, ColumnValueType.STRING)
            .addValue(FINISH_DATE_COLUMN, ColumnValueType.STRING)
            .addValue(FREELANCER_LOGIN_COLUMN, ColumnValueType.STRING)
            .addValue(FREELANCER_LOGIN_UID, ColumnValueType.INT64)
            .addValue(FREELANCER_NAME_COLUMN, ColumnValueType.STRING)
            .build();

    private static final Logger logger = LoggerFactory.getLogger(FreelancersProjectsYtUploader.class);

    private final YtProvider ytProvider;
    private final YtCluster cluster;
    private final String path;

    FreelancersProjectsYtUploader(YtProvider ytProvider, YtCluster cluster) {
        this.ytProvider = ytProvider;
        this.cluster = cluster;
        YtClusterConfig ytClusterConfig = ytProvider.getClusterConfig(cluster);
        this.path =
                YtPathUtil.generatePath(ytClusterConfig.getHome(), relativePart(), FREELANCERS_PROJECTS_EXPORT_PATH);
    }

    void createTableIfAbsent() {
        YtOperator ytOperator = ytProvider.getOperator(cluster);
        Yt yt = ytOperator.getYt();
        YPath table = YPath.simple(path);
        logger.info("cluster: {}, table: {}", cluster, table);

        if (yt.cypress().exists(table)) {
            logger.info("Table already exists");
            return;
        }

        try {
            // создаем динамическую таблицу
            logger.info("Creating table");
            yt.cypress().create(table, CypressNodeType.TABLE, true, false,
                    Cf.map(YtUtils.SCHEMA_ATTR, schema.toYTree(), "dynamic", YTree.booleanNode(true)));

            // монтируем таблицу
            logger.info("Mounting table ..");
            ytOperator.mount(table, 60_000);
            logger.info("Successfully mounted table");

        } catch (Exception e) {
            // при ошибке удаляем созданную таблицу
            try {
                if (yt.cypress().exists(table)) {
                    yt.cypress().remove(table);
                }
            } catch (Exception e1) {
                logger.error("Error in cleanup", e1);
            }
            throw e;
        }
    }

    void upload(List<YtFreelancerProject> freelancersProjects) {
        YtDynamicOperator operator = ytProvider.getDynamicOperator(cluster);

        logger.info("Updating freelancers_projects");
        operator.runInTransaction(transaction -> replaceWithNewData(transaction, freelancersProjects));
        logger.info("Successfully finished updating");
    }

    private void replaceWithNewData(ApiServiceTransaction transaction, List<YtFreelancerProject> freelancersProjects) {
        ModifyRowsRequest insertRequest = new ModifyRowsRequest(path, schema);

        for (YtFreelancerProject project : freelancersProjects) {
            insertRequest.addInsert(Arrays.asList(
                    project.getProjectId(),
                    project.getFreelancerClientId(),
                    project.getCustomerClientId(),
                    dateTimeToString(project.getStartDate()),
                    dateTimeToString(project.getFinishDate()),
                    project.getFreelancerLogin(),
                    project.getFreelancerUid(),
                    project.getFreelancerName()));
        }

        logger.info("Inserting rows");
        transaction.modifyRows(insertRequest).join(); // IGNORE-BAD-JOIN DIRECT-149116
    }

    private static String dateTimeToString(@Nullable LocalDateTime dateTime) {
        return dateTime != null ? dateTime.format(DATE_TIME_FORMATTER) : null;
    }
}
