package ru.yandex.direct.jobs.freelancers;

import java.util.Collection;
import java.util.List;
import java.util.LongSummaryStatistics;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerFeedback;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerUgcModerationStatus;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerRepository;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerFeedbackService;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static ru.yandex.direct.common.db.PpcPropertyNames.ENABLE_UPDATE_FREELANCER_RATINGS;
import static ru.yandex.direct.common.db.PpcPropertyNames.MIN_FREELANCER_FEEDBACKS_COUNT;
import static ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter.enabledFreelancers;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2, minutes = 5),
        needCheck = NonDevelopmentEnvironment.class,
        tags = {DIRECT_PRIORITY_1, JOBS_RELEASE_REGRESSION},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_MAXLOG,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = NonDevelopmentEnvironment.class)
public class UpdateFreelancerRatingsJob extends DirectShardedJob {
    private static final int CHUNK_SIZE = 20;
    private static final int DEFAULT_MIN_FREELANCER_FEEDBACKS_COUNT = 10;
    private static final RelaxedWorker relaxedWorker = new RelaxedWorker(3.0);

    private static final Logger logger = LoggerFactory.getLogger(UpdateFreelancerRatingsJob.class);

    private final FreelancerRepository freelancerRepository;
    private final FreelancerFeedbackService freelancerFeedbackService;
    private final PpcPropertiesSupport ppcPropertiesSupport;


    @Autowired
    public UpdateFreelancerRatingsJob(
            FreelancerRepository freelancerRepository,
            FreelancerFeedbackService freelancerFeedbackService,
            PpcPropertiesSupport ppcPropertiesSupport) {
        this.freelancerRepository = freelancerRepository;
        this.freelancerFeedbackService = freelancerFeedbackService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    /**
     * Конструктор нужен только для тестов. Используется для указания шарда.
     */
    UpdateFreelancerRatingsJob(int shard,
                               FreelancerRepository freelancerRepository,
                               FreelancerFeedbackService freelancerFeedbackService,
                               PpcPropertiesSupport ppcPropertiesSupport) {
        super(shard);
        this.freelancerRepository = freelancerRepository;
        this.freelancerFeedbackService = freelancerFeedbackService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    @Override
    public void execute() {
        if (!isJobEnabled()) {
            logger.info("Skip processing. Job is not enabled.");
            return;
        }
        int shard = getShard();
        int minFeedbacksCount = getMinFeedbacksCount();

        logger.debug("get all freelancerIds");
        Set<Long> allFreelancerIds = getAllFreelancersIds(shard);

        // бьём список фрилансеров на небольшие группы,
        // так как сейчас ходим в UGC DB за каждым фрилансером отдельно
        for (List<Long> chunk : Iterables.partition(allFreelancerIds, CHUNK_SIZE)) {
            logger.debug("update next records chunk");
            logger.trace("Ids for update: {}", chunk);

            relaxedWorker.runAndRelax(() -> processChunk(shard, chunk, minFeedbacksCount));
        }
    }

    private boolean isJobEnabled() {
        return ppcPropertiesSupport.get(ENABLE_UPDATE_FREELANCER_RATINGS).getOrDefault(false);
    }

    private Set<Long> getAllFreelancersIds(int shard) {
        Collection<Freelancer> freelancers = freelancerRepository.getByFilter(shard,
                enabledFreelancers().build());
        return listToSet(freelancers, FreelancerBase::getFreelancerId);
    }

    private int getMinFeedbacksCount() {
        return ppcPropertiesSupport.get(MIN_FREELANCER_FEEDBACKS_COUNT).getOrDefault(DEFAULT_MIN_FREELANCER_FEEDBACKS_COUNT);
    }

    private List<FreelancerFeedback> getFreelancerFeedbackList(List<Long> freelancerIds) {
        return StreamEx.of(freelancerIds)
                .flatCollection(freelancerFeedbackService::getFreelancerFeedbackList)
                .nonNull()
                .toList();
    }

    private void processChunk(int shard, List<Long> freelancerIds, int minFeedbacksCount) {
        List<FreelancerFeedback> feedbacks = getFreelancerFeedbackList(freelancerIds);

        Map<Long, LongSummaryStatistics> statisticsByFreelancerId = StreamEx.of(feedbacks)
                .filter(feedback -> FreelancerUgcModerationStatus.ACCEPTED.equals(feedback.getModerationStatus()))
                .mapToEntry(FreelancerFeedback::getFreelancerId, FreelancerFeedback::getOverallMark)
                .grouping(Collectors.summarizingLong(m -> m));

        Map<Long, Double> ratingByFreelancerId = EntryStream.of(statisticsByFreelancerId)
                .filterValues(s -> s.getCount() >= minFeedbacksCount)
                .mapValues(LongSummaryStatistics::getAverage)
                .mapValues(a -> Math.round(a / 0.5) * 0.5) // округление до ближайшего с шагом 0.5
                .toMap();

        Map<Long, Long> feedbackCountByFreelancerId = EntryStream.of(statisticsByFreelancerId)
                .mapValues(LongSummaryStatistics::getCount)
                .toMap();

        List<Freelancer> freelancers = freelancerRepository.getByIds(shard, freelancerIds);

        List<AppliedChanges<FreelancerBase>> appliedChanges = StreamEx.of(freelancers)
                .map(freelancer -> {
                    Double rating = ratingByFreelancerId.get(freelancer.getId());
                    Long feedbackCount = defaultIfNull(feedbackCountByFreelancerId.get(freelancer.getId()), 0L);
                    return getRatingAppliedChanges(freelancer, rating, feedbackCount);
                })
                .toList();

        freelancerRepository.updateFreelancer(shard, appliedChanges);
    }

    private AppliedChanges<FreelancerBase> getRatingAppliedChanges(FreelancerBase freelancer, Double rating,
                                                                   Long feedbackCount) {
        return new ModelChanges<>(freelancer.getId(), FreelancerBase.class)
                .processNotNull(rating, FreelancerBase.RATING)
                .processNotNull(feedbackCount, FreelancerBase.FEEDBACK_COUNT)
                .applyTo(freelancer);
    }
}
