package ru.yandex.direct.jobs.freelancers.bsratingimport;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtTable;

import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.ENABLE_IMPORT_FREELANCERS_RATING;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Задача, загружающая из YT в базу данных Директа расчитанный ОКР рейтинг фрилансеров
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 12, minutes = 20),
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_MAXLOG,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        ),
        tags = {DIRECT_PRIORITY_2})
@Hourglass(periodInSeconds = /*every 6 hours*/ 6 * 60 * 60, needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class FreelancerBsRatingImportJob extends DirectJob {
    static final String LAST_IMPORTED_TABLE_PROP_NAME =
            String.format("%s_last_imported_table", FreelancerBsRatingImportJob.class.getSimpleName());

    private static final YtCluster IMPORT_CLUSTER = YtCluster.HAHN;
    private static final String INPUT_PATH = FreelancerBsRatingTableRow.FOLDER_PATH;

    private static final Logger logger = LoggerFactory.getLogger(FreelancerBsRatingImportJob.class);

    private final YtCluster ytCluster;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final FreelancerYtRatingService freelancerYtRatingService;
    private final FreelancerUpdateRatingService freelancerUpdateRatingService;

    @Autowired
    public FreelancerBsRatingImportJob(
            PpcPropertiesSupport ppcPropertiesSupport,
            FreelancerYtRatingService freelancerYtRatingService,
            FreelancerUpdateRatingService freelancerUpdateRatingService) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.freelancerYtRatingService = freelancerYtRatingService;
        this.freelancerUpdateRatingService = freelancerUpdateRatingService;
        this.ytCluster = IMPORT_CLUSTER;
    }

    @Override
    public void execute() {
        if (!isJobEnabled()) {
            logger.info("Skip processing. Job is not enabled.");
            return;
        }

        String lastUploadDateTime = fetchNameOfLastImportedTable();

        YtTable table = findTableToImport(lastUploadDateTime);

        if (table == null) {
            logger.info("No fresh source table found. Skip import");
            return;
        }

        String tableForImport = table.getName();
        if (tableForImport.compareTo(LocalDate.now().format(DateTimeFormatter.ISO_LOCAL_DATE)) > 0) {
            logger.warn("Found table from future. Skip import");
            setJugglerStatus(JugglerStatus.WARN, "Source table %s is in future. Seems wrong");
            return;
        }
        logger.info("Importing table {}", table.getPath());

        importFreelancerRatingToDirect(ytCluster, table);

        setNameOfLastImportedTable(tableForImport);
    }

    /**
     * Прочитать рейтинг фрилансеров из таблицы в YT и загрузить их в базу Директа
     */
    void importFreelancerRatingToDirect(YtCluster ytCluster, YtTable table) {
        List<FreelancerBase> ratings = freelancerYtRatingService.readRatingsFromYt(ytCluster, table);
        logger.debug("Update rating for {} freelancers", ratings.size());
        freelancerUpdateRatingService.updateRatingForAll(ratings);
    }

    private boolean isJobEnabled() {
        String propValue = ppcPropertiesSupport.get(ENABLE_IMPORT_FREELANCERS_RATING.getName());
        return propValue != null && propValue.trim().equals("1");
    }

    /**
     * Получить имя последней загруженной из YT таблицы
     */
    @Nullable
    private String fetchNameOfLastImportedTable() {
        String lastImportedTableName = ppcPropertiesSupport.get(LAST_IMPORTED_TABLE_PROP_NAME);
        logger.info("Table import time is {}", lastImportedTableName);
        return lastImportedTableName;
    }

    /**
     * Записать в БД имя последней загруженной из YT таблицы
     */
    private void setNameOfLastImportedTable(String importedTableName) {
        logger.info("Setting prop {} to {}", LAST_IMPORTED_TABLE_PROP_NAME, importedTableName);
        ppcPropertiesSupport.set(LAST_IMPORTED_TABLE_PROP_NAME, importedTableName);
    }

    /**
     * Возвращает последнюю таблицу с рейтингом, если время ее модификации больше времени последнего импорта
     */
    @Nullable
    private YtTable findTableToImport(@Nullable String lastImportedTableName) {
        String freshestTable = freelancerYtRatingService.getFreshestRatingTableName(ytCluster, INPUT_PATH);
        if (freshestTable == null) {
            return null;
        }

        if (lastImportedTableName == null ||
                freshestTable.compareTo(lastImportedTableName) > 0) {
            return getYtTableByRelativePath(freshestTable);
        }

        return null;
    }

    /**
     * По переданному имени вернуть инстанс YT-таблицы
     */
    private YtTable getYtTableByRelativePath(String tableName) {
        return new YtTable(YtPathUtil.generatePath(INPUT_PATH, tableName));
    }

}
