package ru.yandex.direct.jobs.freelancers.bsratingimport;

import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

/**
 * Консьюмер, обрабатывающий строки с рейтингом фрилансеров, выгружаемым ОКР в YT.
 */
@ParametersAreNonnullByDefault
class FreelancerBsRatingRowConsumer implements Consumer<FreelancerBsRatingTableRow> {
    private static final Logger logger = LoggerFactory.getLogger(FreelancerBsRatingRowConsumer.class);

    private final long expectedTotalRowCount;
    private final long startTime;

    private long rowsProcessed;
    private final Map<Long, FreelancerRating> ratingByClientID;


    FreelancerBsRatingRowConsumer(int expectedTotalRowCount) {
        this.expectedTotalRowCount = expectedTotalRowCount;
        this.startTime = getTimestampSeconds();

        rowsProcessed = 0;
        ratingByClientID = new HashMap<>(expectedTotalRowCount);
    }

    private static long getTimestampSeconds() {
        return System.currentTimeMillis() / 1000L;
    }

    private long getSecondsLeft(long timestamp) {
        if (rowsProcessed == 0) {
            return 0;
        }
        double timeWorking = timestamp - (double) startTime;
        return Math.round(timeWorking / rowsProcessed * expectedTotalRowCount);
    }

    /**
     * Подготовить собранные данные и отправить их следующему потребителю
     */
    List<FreelancerBase> getDataAndCleanup() {
        logger.info("Returning data for {} freelancers", ratingByClientID.size());
        long timestamp = getTimestampSeconds();
        logger.info(
                "Processed {} rows of {}, importing for {} seconds, approx {} seconds left",
                rowsProcessed, expectedTotalRowCount, Duration.ofSeconds(timestamp - startTime),
                Duration.ofSeconds(getSecondsLeft(timestamp)));

        if (ratingByClientID.isEmpty()) {
            return Collections.emptyList();
        }

        List<FreelancerBase> forecasts = ratingByClientID.entrySet().stream()
                .map(e -> new FreelancerBase()
                        .withId(e.getKey())
                        .withAdvQualityRating(e.getValue().getRating())
                        .withAdvQualityRank(e.getValue().getRank())
                )
                .collect(Collectors.toList());

        ratingByClientID.clear();
        return forecasts;
    }

    @Override
    public void accept(FreelancerBsRatingTableRow tableRow) {
        rowsProcessed++;

        long clientId = tableRow.getClientId();
        String ratingKey = tableRow.getRatingKey();
        YTreeNode flRatingValue = tableRow.getRatingDict().get(ratingKey);
        if (flRatingValue == null) {
            logger.warn("Missed rating by key {} for freelancer with id {} (row {})",
                    ratingKey, clientId, rowsProcessed);
            return;
        }

        FreelancerRating rating = new FreelancerRating()
                .setRank(tableRow.getFreelancerRank())
                .setUnscaledRating(flRatingValue.doubleValue());

        ratingByClientID.put(clientId, rating);
    }
}
