package ru.yandex.direct.jobs.freelancers.bsratingimport;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerUpdateService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.Result;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter.allFreelancersIncludingDisabled;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class FreelancerUpdateRatingService {
    private static final Logger logger = LoggerFactory.getLogger(FreelancerUpdateRatingService.class);

    private final FreelancerService freelancerService;
    private final FreelancerUpdateService freelancerUpdateService;

    public FreelancerUpdateRatingService(
            FreelancerService freelancerService,
            FreelancerUpdateService freelancerUpdateService) {
        this.freelancerService = freelancerService;
        this.freelancerUpdateService = freelancerUpdateService;
    }

    /**
     * Обновляет рейтинги для <i>всех</i> специалистов.
     */
    void updateRatingForAll(Collection<FreelancerBase> ratings) {
        List<Long> freelancerIds = mapList(ratings, FreelancerBase::getId);
        FreelancersQueryFilter allFreelancersFilter =
                allFreelancersIncludingDisabled().withFreelancerIds(freelancerIds).build();
        List<Freelancer> existingFreelancers = freelancerService.getFreelancers(allFreelancersFilter);
        Set<Long> existingFreelancerIds = listToSet(existingFreelancers, FreelancerBase::getId);

        Collection<FreelancerBase> ratingsToUpdate = ratings.stream()
                .filter(change -> existingFreelancerIds.contains(change.getId()))
                .collect(toList());
        if (ratingsToUpdate.size() < ratings.size()) {
            logger.info("Can't update rating for {} missed freelancers", ratings.size() - ratingsToUpdate.size());
        }

        logger.debug("Update freelancers: {}", ratingsToUpdate);
        // todo maxlog: подумать о bulk'овом обновлении.
        for (FreelancerBase freelancerUpdateRequest : ratingsToUpdate) {
            ClientId clientId = ClientId.fromLong(freelancerUpdateRequest.getId());
            Result<Long> result = freelancerUpdateService.updateFreelancer(clientId, freelancerUpdateRequest);
            if (!result.isSuccessful()) {
                // todo maxlog: нужно при ошибках зажигать мониторинг
                logger.error("Unexpected error on updating freelancer rating: {}",
                        result.getValidationResult());
            }
        }
    }

}
