package ru.yandex.direct.jobs.freelancers.bsratingimport;

import java.util.List;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtOperator;
import ru.yandex.direct.ytwrapper.model.YtTable;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;

import static com.google.common.primitives.Ints.checkedCast;

@Service
public class FreelancerYtRatingService {
    private static final Pattern DATE_PATTERN = Pattern.compile("\\d{4}-\\d{2}-\\d{2}");

    private final YtProvider ytProvider;
    private final FreelancerBsRatingRowConsumerProvider rowConsumerProvider;

    @Autowired
    public FreelancerYtRatingService(YtProvider ytProvider,
                                     FreelancerBsRatingRowConsumerProvider rowConsumerProvider) {
        this.ytProvider = ytProvider;
        this.rowConsumerProvider = rowConsumerProvider;
    }

    /**
     * Получить имя таблицы из {@code ytPath} с наиболее свежими данными.
     * Свежесть определяется по имени таблицы.
     * Учитываются только таблицы, чьи имена подпадают по паттерн {@link #DATE_PATTERN}
     */
    @Nullable
    String getFreshestRatingTableName(
            YtCluster ytCluster,
            @SuppressWarnings("SameParameterValue") String ytPath) {
        List<YTreeStringNode> nodes = ytProvider.get(ytCluster).cypress().list(YPath.simple(ytPath));
        return StreamEx.of(nodes)
                .map(YTreeNode::stringValue)
                .filter(s -> DATE_PATTERN.matcher(s).matches())
                .reverseSorted()
                .findFirst()
                .orElse(null);
    }

    /**
     * Возвращает рейтинги специалистов в виде запросов на обновление {@link FreelancerBase}
     */
    List<FreelancerBase> readRatingsFromYt(YtCluster ytCluster, YtTable table) {
        YtOperator ytOperator = ytProvider.getOperator(ytCluster);
        int rowCount = checkedCast(ytOperator.readTableNumericAttribute(table, "row_count"));
        FreelancerBsRatingRowConsumer rowConsumer = rowConsumerProvider.getConsumer(rowCount);

        return getRatingsFromYt(ytOperator, table, rowConsumer);
    }

    private List<FreelancerBase> getRatingsFromYt(
            YtOperator ytOperator,
            YtTable table,
            FreelancerBsRatingRowConsumer rowConsumer) {
        // Пока что количество фрилансеров меньше 1_000, считаем допустимым вычитывать всё в память
        ytOperator.readTable(table, new FreelancerBsRatingTableRow(), rowConsumer);
        return rowConsumer.getDataAndCleanup();
    }

}
