package ru.yandex.direct.jobs.freelancers.certimport;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.FREELANCER_CERTIFICATES_UPDATING_ENABLED;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Джоба обновляющая сертификаты фрилансеров.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 120 + 5),
        needCheck = NonDevelopmentEnvironment.class,
        tags = {DIRECT_PRIORITY_2, JOBS_RELEASE_REGRESSION},
        notifications = @OnChangeNotification(
        recipient = NotificationRecipient.LOGIN_MAXLOG,
        method = NotificationMethod.TELEGRAM,
        status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}))
@Hourglass(periodInSeconds = 3600, needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class FreelancerCertificatesUpdateJob extends DirectShardedJob {
    static final String LAST_TIME_PROPERTY_KEY_PATTERN = "fl_cert_last_success_update_time_shard_%d";
    private static final Logger logger = LoggerFactory.getLogger(FreelancerCertificatesUpdateJob.class);
    private static final Duration MIN_EXECUTE_INTERVAL = Duration.ofHours(23);
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final FreelancerCertificatesUpdateService freelancerCertificatesUpdateService;

    @Autowired
    public FreelancerCertificatesUpdateJob(
            PpcPropertiesSupport ppcPropertiesSupport,
            FreelancerCertificatesUpdateService freelancerCertificatesUpdateService) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.freelancerCertificatesUpdateService = freelancerCertificatesUpdateService;
    }

    @Override
    public void execute() {
        String jobEnabledPropValue = ppcPropertiesSupport.get(FREELANCER_CERTIFICATES_UPDATING_ENABLED.getName());
        boolean jobEnabled = Boolean.parseBoolean(jobEnabledPropValue);
        if (!jobEnabled) {
            logger.info("Skip processing. Job is not enabled. {}={}",
                    FREELANCER_CERTIFICATES_UPDATING_ENABLED,
                    jobEnabledPropValue);
            return;
        }
        //Проверяем, прошло ли достаточно времени с последнего обновления сертификатов.
        int shard = getShard();
        String lastSuccessTimeKey = String.format(LAST_TIME_PROPERTY_KEY_PATTERN, shard);
        String propertyValue = ppcPropertiesSupport.get(lastSuccessTimeKey);
        long lastTimeMillis = isNotBlank(propertyValue) ? Long.parseLong(propertyValue) : 0L;
        if ((System.currentTimeMillis() - lastTimeMillis < MIN_EXECUTE_INTERVAL.toMillis())) {
            logger.info("Skip processing. It is too early. {}={}",
                    lastSuccessTimeKey,
                    propertyValue);
            return;
        }
        updateCertificates(shard);
        logger.info("All certificates on shard_{} were updated.", shard);
        //Сохраняем время обновления сертификатов.
        String currentTimeMillisAsString = Long.toString(System.currentTimeMillis());
        ppcPropertiesSupport.set(lastSuccessTimeKey, currentTimeMillisAsString);
    }

    private void updateCertificates(int shard) {
        freelancerCertificatesUpdateService.updateCertificates(shard);
    }

}
