package ru.yandex.direct.jobs.freelancers.moderation;

import java.nio.charset.StandardCharsets;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.kikimr.persqueue.LogbrokerClientFactory;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.compression.CompressionCodec;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.kikimr.persqueue.consumer.sync.SyncConsumerConfig;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;
import ru.yandex.kikimr.persqueue.producer.async.AsyncProducerConfig;
import ru.yandex.kikimr.persqueue.producer.transport.message.inbound.ProducerInitResponse;
import ru.yandex.kikimr.persqueue.proxy.ProxyBalancer;

import static java.util.Collections.singletonList;

@Service
public class LogbrokerClientFactoryService {
    private final TvmIntegration tvmIntegration;

    @Autowired
    public LogbrokerClientFactoryService(TvmIntegration tvmIntegration) {
        this.tvmIntegration = tvmIntegration;
    }

    /**
     * Создаваемый продюсер нужно явно закрывать для освобожнения ресурсов.
     * Он реализует интерфейс AutoCloseable, поэтому можно это делать через конструкцию try().
     */
    public AsyncProducer createProducer(LogbrokerProducerProperties producerProperties, String sourceId)
            throws ExecutionException, TimeoutException {
        try {
            TvmService tvmService = producerProperties.getTvmService();
            Supplier<Credentials> logBrokerCredentialsSupplier = getLogBrokerCredentialsSupplier(tvmService);
            String writeTopic = producerProperties.getWriteTopic();
            AsyncProducerConfig producerConfig = getProducerConfig(logBrokerCredentialsSupplier, writeTopic, sourceId);
            String host = producerProperties.getHost();
            ProxyBalancer proxyBalancer = new ProxyBalancer(host);
            LogbrokerClientFactory lcf = new LogbrokerClientFactory(proxyBalancer);
            AsyncProducer asyncProducer = lcf.asyncProducer(producerConfig);
            //инициализируем сессию
            CompletableFuture<ProducerInitResponse> init = asyncProducer.init();
            init.get(producerProperties.getTimeoutSec(), TimeUnit.SECONDS);
            return asyncProducer;
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
    }

    private Supplier<Credentials> getLogBrokerCredentialsSupplier(TvmService tvmService) {
        return () -> {
            String serviceTicket = tvmIntegration.getTicket(tvmService);
            return Credentials.tvm(serviceTicket);
        };
    }

    private AsyncProducerConfig getProducerConfig(Supplier<Credentials> logBrokerCredentialsSupplier,
                                                  String writeTopic,
                                                  String sourceId) {
        byte[] sourceIdBytes = sourceId.getBytes(StandardCharsets.UTF_8);
        AsyncProducerConfig.Builder builder = AsyncProducerConfig
                .builder(writeTopic, sourceIdBytes)
                .setCodec(CompressionCodec.RAW)
                .setCredentialsProvider(logBrokerCredentialsSupplier);
        return builder.build();
    }

    /**
     * Создаваемый консьюмер нужно явно закрывать для освобожнения ресурсов.
     * Он реализует интерфейс AutoCloseable, поэтому можно это делать через конструкцию try().
     */
    public SyncConsumer createConsumer(LogbrokerConsumerProperties consumerProperties)
            throws TimeoutException {
        try {
            TvmService tvmService = consumerProperties.getTvmService();
            Supplier<Credentials> logBrokerCredentialsSupplier = getLogBrokerCredentialsSupplier(tvmService);
            SyncConsumerConfig consumerConfig = getConsumerConfig(consumerProperties, logBrokerCredentialsSupplier);
            String host = consumerProperties.getHost();
            ProxyBalancer proxyBalancer = new ProxyBalancer(host);
            LogbrokerClientFactory lcf = new LogbrokerClientFactory(proxyBalancer);
            long timeoutSec = consumerProperties.getTimeoutSec();
            SyncConsumer syncConsumer;
            syncConsumer = lcf.syncConsumer(consumerConfig, timeoutSec, TimeUnit.SECONDS);

            syncConsumer.init();
            return syncConsumer;
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
    }

    private SyncConsumerConfig getConsumerConfig(LogbrokerConsumerProperties consumerProperties,
                                                 Supplier<Credentials> logBrokerCredentialsSupplier) {
        String readTopic = consumerProperties.getReadTopic();
        String consumerName = consumerProperties.getConsumerName();
        int timeoutSec = (int) consumerProperties.getTimeoutSec();
        return SyncConsumerConfig.builder(singletonList(readTopic), consumerName)
                .setReadDataTimeout(timeoutSec, TimeUnit.SECONDS)
                .setInitTimeout(timeoutSec, TimeUnit.SECONDS)
                .setCredentialsProvider(logBrokerCredentialsSupplier)
                .build();
    }

}
