package ru.yandex.direct.jobs.freelancers.moderation.receiving;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeoutException;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.jobs.freelancers.moderation.LogbrokerClientFactoryService;
import ru.yandex.direct.jobs.freelancers.moderation.LogbrokerConsumerProperties;
import ru.yandex.direct.jobs.freelancers.moderation.receiving.model.VerdictType;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;

import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.FREELANCER_CARDS_MODERATION_RECEIVING_ENABLED;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;

/**
 * Джоба принимающая вердикт модерации на карточку фрилансера.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 20 + 1),
        needCheck = NonDevelopmentEnvironment.class,
        tags = {DIRECT_PRIORITY_1, JOBS_RELEASE_REGRESSION},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_MAXLOG,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 600, needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class FreelancerCardReceivingJob extends DirectJob {
    public static final String CONFIG_SECTION_NAME = "fl_card_lb_moderation_receiving";
    private static final Logger logger = LoggerFactory.getLogger(FreelancerCardReceivingJob.class);
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final LogbrokerConsumerProperties consumerProperties;
    private final LogbrokerClientFactoryService logbrokerClientFactoryService;
    private final Map<VerdictType, VerdictHandler> verdictHandlersByVerdictType = new HashMap<>();

    @Autowired
    public FreelancerCardReceivingJob(
            PpcPropertiesSupport ppcPropertiesSupport,
            DirectConfig directConfig,
            LogbrokerClientFactoryService logbrokerClientFactoryService,
            FreelancerCardVerdictHandler freelancerCardVerdictHandler) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.consumerProperties = LogbrokerConsumerProperties.createInstance(directConfig, CONFIG_SECTION_NAME);
        this.logbrokerClientFactoryService = logbrokerClientFactoryService;
        addVerdictHandler(freelancerCardVerdictHandler);
    }

    private void addVerdictHandler(VerdictHandler verdictHandler) {
        verdictHandlersByVerdictType.put(verdictHandler.getSupportedType(), verdictHandler);
    }


    @Override
    public void execute() {
        String jobEnabledPropValue = ppcPropertiesSupport.get(FREELANCER_CARDS_MODERATION_RECEIVING_ENABLED.getName());
        boolean jobEnabled = Boolean.parseBoolean(jobEnabledPropValue);
        if (!jobEnabled) {
            logger.info("Skip processing. Job is not enabled. {}={}",
                    FREELANCER_CARDS_MODERATION_RECEIVING_ENABLED,
                    jobEnabledPropValue);
            return;
        }
        readVerdicts();
    }

    private void readVerdicts()  {
        VerdictReceiver verdictReceiver = null;
        try (SyncConsumer syncConsumer = logbrokerClientFactoryService.createConsumer(consumerProperties)) {
            verdictReceiver = new VerdictReceiver(syncConsumer, verdictHandlersByVerdictType);
            verdictReceiver.readAndApply();
        } catch (RuntimeException | TimeoutException ex) {
            throw new RuntimeException("Error while reading a freelancer card moderation verdicts from LogBroker",
                    ex);
        } finally {
            if (verdictReceiver != null) {
                logger.info("Lb messages were received: {}, Verdicts were received {}, Verdicts were applied: {}",
                        verdictReceiver.getLbMessageCount(),
                        verdictReceiver.getReceivedVerdictCount(),
                        verdictReceiver.getAppliedVerdictCount());
            }
        }
    }
}
