package ru.yandex.direct.jobs.freelancers.moderation.receiving;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.freelancer.model.FreelancerCardModeration;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerCardService;
import ru.yandex.direct.jobs.freelancers.moderation.receiving.model.CardModerationVerdict;
import ru.yandex.direct.jobs.freelancers.moderation.receiving.model.VerdictType;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.result.Defect;

@Component
@ParametersAreNonnullByDefault
public class FreelancerCardVerdictHandler implements VerdictHandler<CardModerationVerdict> {

    private static final Logger logger = LoggerFactory.getLogger(FreelancerCardVerdictHandler.class);
    private final FreelancerCardService freelancerCardService;

    @Autowired
    public FreelancerCardVerdictHandler(FreelancerCardService freelancerCardService) {
        this.freelancerCardService = freelancerCardService;
    }

    @Override
    public boolean applyVerdict(String jsonVerdict) {
        FreelancerCardModeration moderationResult;
        try {
            CardModerationVerdict cardModerationVerdict = JsonUtils.fromJson(jsonVerdict, CardModerationVerdict.class);
            moderationResult = Converter.createModeration(cardModerationVerdict);
        } catch (IllegalArgumentException ex) {
            logger.error("Parsing jsonVerdict to CardModerationVerdict ended with error: {} Verdict info: {}", ex,
                    jsonVerdict);
            return false;
        }
        Result<Long> result = freelancerCardService.applyModerationResult(moderationResult);
        if (result.getState().equals(ResultState.SUCCESSFUL)) {
            return true;
        }
        List<Defect> defects = result.getValidationResult().getErrors();
        String defectsDescription = StreamEx.of(defects)
                .map(Defect::toString)
                .joining("; ");
        logger.error("In FreelancerCardModeration were found defects: {}. Verdict info: {}", defectsDescription,
                jsonVerdict);
        return false;
    }

    @Override
    public VerdictType getSupportedType() {
        return VerdictType.FREELANCER_CARD;
    }

}
